// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com

//  gen.js
//  Revised 2018-06-11
//
//  This script creates a series of 36-board deal sets,
//  Morning, Afternoon, and Evening for each day in a specified range.
//  Each deal set consists of a PBN and a PDF file.

//  The hand record files are formatted like the file
//  BCHRLayout.pbn in the user's Documents folder.
//  If that file does not exist, it is created.

//  The Documents folder sub-folder to receive the generated deal sets:
var folder = 'gen';

var bc = WScript.CreateObject('BridgeComposer.Object');

//  This array lists the sessions to be generated for each date in the range.
//      Note that the 1st character of each session name will be used
//      as the 7th character of each filename generated.
var session = new Array('Morning Pairs', 'Afternoon Pairs', 'Evening Pairs');

function pad(n)
{
  return n < 10 ? '0' + n : '' + n;
}

var dateFormat = 0;    // select date format (0, 1, or 2: see formatDate function)
function formatDate(dt) {
  switch (dateFormat) {
  case 0:    //  m/d/yyyy
  default:
    var s = (dt.getMonth() + 1) + '/';
    s += dt.getDate() + '/';
    s += dt.getFullYear();
    return s;
    
  case 1:    //  d/m/yyyy
    var s = dt.getDate() + '/';
    s += (dt.getMonth() + 1) + '/';
    s += dt.getFullYear();
    return s;
    
  case 2:    //  yyyy-mm-dd
    var s = dt.getFullYear() + '-';
    s += pad(dt.getMonth() + 1) + '-';
    s += pad(dt.getDate());
    return s;
  }
}

function promptDate(text, defDate) {
  var pt = 'gen.js: Generate deal sets\n' + text;
  if (!defDate) {
    // get today at midnight
    var now = new Date();
    defDate = new Date(now.getFullYear(), now.getMonth(), now.getDate());
  }
  var pd = formatDate(defDate);
  while (true) {
    var s = bc.prompt(pt, pd);
    if (!s) WScript.Quit();  // user Cancel
    var dt;
    switch (dateFormat) {
    case 0:    //  m/d/yyyy (or see JS Date.parse documentation for other allowable formats)
    default:
      dt = new Date(s);
      break;
      
    case 1:    //  d/m/yyyy
      var ar = s.split('/');
      if (ar.length === 3)
        dt = new Date(ar[2], ar[1] - 1, ar[0]);
      else
        dt = new Date(s);
      break;
      
    case 2:    //  yyyy-mm-dd
      var ar = s.split('-');
      if (ar.length == 3)
        dt = new Date(ar[0], ar[1] - 1, ar[2]);
      else
        dt = new Date(s);
      break;
    }
    
    //  check for date within BC range (1970-01-01 to 3000-12-31)
    if (dt.getTime()
      && dt.getTime() >= (new Date(1970, 0, 1)).getTime()
      && dt.getTime() <= (new Date(3000, 11, 31)).getTime())      
      return dt;  // value is valid and in-range
      
    bc.alert('An invalid date was entered. Try again.', 16);
  }
}

var wsh = WScript.CreateObject('WScript.Shell');
var pathDocs = wsh.SpecialFolders('MyDocuments');
var pathLayout = pathDocs + '\\BCHRLayout.pbn';
var fso = WScript.CreateObject('Scripting.FileSystemObject');
if (!fso.FileExists(pathLayout)) {
  var s = bc.prompt('gen.js: Generate deal sets\nEnter your "Site" title', 'Our Bridge Club');
  if (!s) WScript.Quit();
  var ts = fso.CreateTextFile(pathLayout);
  ts.WriteLine('%BCOptions Center GutterH GutterV STBorder STShade');
  ts.WriteLine('%BoardsPerPage 18');
  ts.WriteLine('%Margins 252,250,250,250');
  ts.WriteLine('%HRTitleSite "' + s + '"');
  ts.Close();
}
bc.Open(pathLayout);
var strSite = bc.Site;
var pathFolder = pathDocs + '\\' + folder;
if (!fso.FolderExists(pathFolder)) fso.CreateFolder(pathFolder);
var first = promptDate('Enter starting date');
var last = promptDate('Enter ending date', first);
var ndays = (last.getTime() - first.getTime()) / 86400000 + 1;
if (ndays < 0) ndays = 0;
var plural = (ndays === 1) ? '' : 's';
if (!bc.confirm('Generate boards for '
  + formatDate(first) + ' through ' + formatDate(last)
  + ' (' + ndays + ' day' + plural + ')?', 64)) {
  WScript.Quit();
}
var count = 0;
for (var date = first; date <= last; date.setDate(date.getDate() + 1)) {
  for (var s = 0; s < session.length; ++s) {
    bc.New();
    bc.LoadLayout(pathLayout);
    bc.Site = strSite;
    bc.Event = session[s];
    bc.Date = date.getVarDate();
    bc.GenerateRandomBoards(36);

    var y = pad(date.getFullYear() % 100);
    var m = pad(date.getMonth() + 1);
    var d = pad(date.getDate());
    var x = session[s].slice(0,1);
    var filename = y + m + d + x;
    
    var pathFile = pathFolder + '\\' + filename + '.pbn';
    if (fso.FileExists(pathFile)) {
      var msg = 'The file ' + pathFile + ' already exists.';
      msg += ' It will not be overwritten.';
      if (bc.confirm(msg, 16)) continue;
      else WScript.Quit();
    }

    // Save now, so output filename will show in DDA progress panel
    bc.SaveAs(pathFile);
    bc.DoubleDummyAllBoards();
    bc.Save();  // Save again, with DDA included
    bc.SaveAsPdf(pathFolder + '\\' + filename + '.pdf');
    ++count;
     }
}
var plural = (count === 1) ? '' : 's';
bc.alert('gen.js: Generated ' + count + ' deal set' + plural
  + ' in folder ' + pathFolder + '.', 64);
