// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com

//  This script extracts score data from all the boards in a bridge document,
//  calculates the matchpoints, and outputs the data to a spreadsheet
//  CSV (comma-separated values) file.

//  $Id: TabulateScoresMP.js 219 2024-11-13 19:09:04Z Ray $


var bc = WScript.CreateObject('BridgeComposer.Object');
if (WScript.Arguments.length > 0) {
  bc.Open(WScript.Arguments(0));
} else {
  if (!bc.Open()) WScript.Quit();
}


var CLUBS = 0;
var DIAMS = 1;
var HEARTS = 2;
var SPADES = 3;
var NOTRUMP = 4;

function CalcScore(nLevel, nStrain, nRisk, bVul, nResult)
{
  //  nLevel = bid level (1..7) or 0 for pass-out
  //  nStrain = bid suit or notrump (see above)
  //  nRisk = 0 (undoubled), 1 (doubled), or 2 (redoubled)
  //  bVul = "true" if declarer is vulnerable
  //  nResult = tricks taken by declarer (0..13)
  
  if (nLevel === 0)
    return 0;
  
  var nScore = 0;
  if (nResult < nLevel + 6) {
    var nDown = nLevel + 6 - nResult;
    nScore = (bVul) ? -100 : -50;
    if (nRisk === 0) {
      nScore *= nDown;
    } else {
      // doubled (or redoubled)
      nScore *= 2;
      if (bVul) {
        if (nDown > 1)
          nScore -= 300 * (nDown - 1);
      } else {
        if (nDown > 1)
          nScore -= 200 * (nDown - 1);
        
        if (nDown > 3)
          nScore -= 100 * (nDown - 3);
      }
      
      if (nRisk === 2)
        nScore *= 2;  // redoubled
    }
  } else {
    var nOver = nResult - 6 - nLevel;
    var nPerTrick;
    switch (nStrain) {
    case NOTRUMP:
      nScore = 10;
      // FALL-THRU
    case SPADES:
    case HEARTS:
      nPerTrick = 30;
      break;
    
    case DIAMS:
    case CLUBS:
      nPerTrick = 20;
      break;
    }
    
    nScore += nLevel * nPerTrick; // contract score
    if (nRisk > 0)
      nScore *= 2;  // doubled
    
    if (nRisk === 2)
      nScore *= 2;  // redoubled
    
    if (nScore < 100)
      nScore += 50; // part-score bonus
    else
      nScore += (bVul) ? 500 : 300; // game bonus
    
    if (nLevel === 6)
      nScore += (bVul) ? 750 : 500; // small slam bonus
    else if (nLevel === 7)
      nScore += (bVul) ? 1500 : 1000; // grand slam bonus
    
    nScore += 50 * nRisk; // for the insult (if any)
    
    var nPerOver = nPerTrick;
    if (nRisk) {
      nPerOver = (bVul) ? 200 : 100;
      if (nRisk > 1)
        nPerOver *= 2;
    }
    
    nScore += nOver * nPerOver; // overtricks
  }
  
  return nScore;
}


function GetScore(obj)
{
  var brd = obj.board;
  
  //  Contract
  
  var strContract = brd.TagValue('Contract');
  
  //  Result
  
  var strResult = brd.TagValue('Result');
  
  //  Per the PBN standard, a pass-out should have
  //  Contract=="Pass" and Result=="", but we have seen
  //  Contract="" and Result="0".
  //  So we interpret both the above as a pass-out, otherwise
  //  we interpret Result="" as a no-play.
  
  if (strContract === '' && strResult === '0') {
    strContract = 'Pass';
    strResult = '';
  }
  
  if (strContract !== 'Pass' && strResult === '') {
    strContract = '';   // unfinished board, treat as no-play
  }

  //  Vulnerability

  var strVul = brd.TagValue('Vulnerable');
  if (strVul === 'All')
    strVul = 'Both';

  obj.strVul = strVul;
  obj.strContract = strContract;
  obj.strResult = strResult;
  obj.strMake = '';
  obj.strDeclarer = '';

  if (strContract !== 'Pass' && strResult === '')
    return 0;
  
  var nLevel = 0;
  var nStrain = NOTRUMP;
  var nRisk = 0;
  if (strContract !== 'Pass') {
    nLevel = parseInt(strContract.charAt(0));
    nStrain = 'CDHSN'.indexOf(strContract.charAt(1));
    for (var i = 2; i < strContract.length; ++i) {
      var ch = strContract.charAt(i);
      if (ch === 'X')
        ++nRisk;
    }
  }
  
  var nResult = parseInt(strResult);
  if (isNaN(nResult))
    nResult = 0;

  var strMake;
  if (nLevel > 0) {
    strMake = nResult - (nLevel + 6);
    if (strMake === 0)
      strMake = '"= "';
  } else {
    strMake = nResult;
    if (strMake === 0)
      strMake = '';
  }

  obj.strMake = '' + strMake;
      
  //  Declarer
  
  strDeclarer = brd.TagValue('Declarer');
  var bNS = (strDeclarer === 'N' || strDeclarer === 'S');
  obj.strDeclarer = strDeclarer;
  
  //  vulnerability of declarer
    
  var bVul = false;
  switch (strVul) {
  case 'None':
    break;
    
  case 'NS':
    bVul = bNS;
    break;
  
  case 'EW':
    bVul = !bNS;
    break;
    
  case 'Both':
    bVul = true;
    break;
  }
  
  //  Score

  var nScore = CalcScore(nLevel, nStrain, nRisk, bVul, nResult);      
  if (!bNS)
    nScore = -nScore;

  return nScore;
}


function CalcMP(vBoard)
{
  //  Call with vBoard = [].
  //  vBoard will be populated as an array of arrays.
  //  The first index is the board number minus 1.
  //  Each of these elements is an array of objects, one object for each
  //  instance of the same board number.
  //  Each object will have members "score", "mpns", and "mpew".
  //  "score" is the bridge score on the board, positive for NS and negative for EW.
  //  "mpns" and "mpew" are the matchpoints for NS and EW respectively.
  //  Matchpoints are calculated by PBN "MP1" method, that is,
  //  2 MP for each lower score and 1 MP for each equal score.

  var bds = bc.Boards;
  while (bds.MoveNext()) {
    var bd = bds.Current;
    var bn = parseInt(bd.TagValue('Board'));
    if (bn < 1)
      continue;

    --bn;
    if (vBoard[bn] === undefined)
      vBoard[bn] = [];

    var be = vBoard[bn];
    var score = GetScore({board: bd});
    be.push({score: score})
  }

  //  Determine number of plays, which determines top matchpoint score
  //  Display raw data: board number and existing bridge scores

  var nPlays = -1;
  var str = '';
  for (var i = 0; i < vBoard.length; ++i) {
    var be = vBoard[i];
    if (be != undefined) {
      if (nPlays < 0)
        nPlays = be.length;
      else if (be.length !== nPlays) {
        if (!bc.confirm('Board ' + (i + 1) + ': ' + be.length + ' plays (not ' + nPlays + ')'))
          WScript.Quit();
      }

      str += (i + 1);
      str += ' (' + be.length + ')';
      for (var j = 0; j < be.length; ++j) {
        str += ' ' + be[j].score;
      }

      str += '\n';
    }
  }

  var bVerbose = false;
  if (bVerbose && !bc.confirm(str))
    WScript.Quit();

  //  calculate matchpoints

  var nTop = (nPlays - 1) * 2;
  for (var i = 0; i < vBoard.length; ++i) {
    var be = vBoard[i];
    if (be != undefined) {
      for (var j = 0; j < be.length; ++j) {
        be[j].mpns = 0;
        for (var k = 0; k < be.length; ++k) {
          if (k !== j) {
            if (be[j].score > be[k].score)
              be[j].mpns += 2;
            else if (be[j].score === be[k].score)
              be[j].mpns += 1;
          }
        }
      }

      for (j = 0; j < be.length; ++j) {
        be[j].mpew = nTop - be[j].mpns;
      }
    }
  }
}


//  Create the CSV output file

var strPathname = bc.BrowseForFile('TabulateScoresMP Output File', 1,
  'CSV Files (*.csv)|*.csv||');
if (!strPathname)
  WScript.Quit();

var fso = WScript.CreateObject('Scripting.FileSystemObject');
var fout = fso.CreateTextFile(strPathname, true);
fout.WriteLine('Deal,Contract,Vuln,By,Result,"Score NS","Score EW",'
  + '"MP NS","MP EW"');

var vMP = [];
CalcMP(vMP);

var cBoards = bc.Boards.Count;
var nBoardIndex = -999;
var nBoardInstance = 0;
for (var iBoard = 0; iBoard < cBoards; ++iBoard) {
  var brd = bc.Boards.Item(iBoard);
  var nx = parseInt(brd.TagValue('Board')) - 1;
  if (nx !== nBoardIndex) {
    nBoardIndex = nx;
    nBoardInstance = 0;
  } else {
    nBoardInstance++;
  }
    
  var strBoard = brd.UniqueBoard;  
  var strScoreNS = '';
  var strScoreEW = '';
  var strMPNS = '';
  var strMPEW = '';
  
  var obj = {board: brd};
  var nScore = GetScore(obj);

  if (nScore >= 0)
    strScoreNS = nScore;
  else
    strScoreEW = -nScore;

  // Matchpoints

  if (vMP[nBoardIndex] !== undefined) {
    var bdata = vMP[nBoardIndex][nBoardInstance];
    strMPNS = bdata.mpns;
    strMPEW = bdata.mpew;
  }

  strLine = strBoard;
  strLine += ',' + obj.strContract;
  strLine += ',' + obj.strVul;
  strLine += ',' + obj.strDeclarer;
  strLine += ',' + obj.strMake;
  strLine += ',' + strScoreNS;
  strLine += ',' + strScoreEW;
  strLine += ',' + strMPNS;
  strLine += ',' + strMPEW;
  
  fout.WriteLine(strLine);
}
