// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com

//  This script extracts score data from all the boards in a bridge document,
//  calculates the "Par IMPS", and outputs the data to a spreadsheet
//  CSV (comma-separated values) file.

//  $Id: TabulateScores.js 183 2024-03-15 21:38:08Z Ray $

var bc = WScript.CreateObject('BridgeComposer.Object');
if (WScript.Arguments.length > 0) {
  bc.Open(WScript.Arguments(0));
} else {
  if (!bc.Open()) WScript.Quit();
}

var CLUBS = 0;
var DIAMS = 1;
var HEARTS = 2;
var SPADES = 3;
var NOTRUMP = 4;

function CalcScore(nLevel, nStrain, nRisk, bVul, nResult)
{
  //  nLevel = bid level (1..7) or 0 for pass-out
  //  nStrain = bid suit or notrump (see above)
  //  nRisk = 0 (undoubled), 1 (doubled), or 2 (redoubled)
  //  bVul = "true" if declarer is vulnerable
  //  nResult = tricks taken by declarer (0..13)
  
  if (nLevel === 0)
    return 0;
  
  var nScore = 0;
  if (nResult < nLevel + 6) {
    var nDown = nLevel + 6 - nResult;
    nScore = (bVul) ? -100 : -50;
    if (nRisk === 0) {
      nScore *= nDown;
    } else {
      // doubled (or redoubled)
      nScore *= 2;
      if (bVul) {
        if (nDown > 1)
          nScore -= 300 * (nDown - 1);
      } else {
        if (nDown > 1)
          nScore -= 200 * (nDown - 1);
        
        if (nDown > 3)
          nScore -= 100 * (nDown - 3);
      }
      
      if (nRisk === 2)
        nScore *= 2;  // redoubled
    }
  } else {
    var nOver = nResult - 6 - nLevel;
    var nPerTrick;
    switch (nStrain) {
    case NOTRUMP:
      nScore = 10;
      // FALL-THRU
    case SPADES:
    case HEARTS:
      nPerTrick = 30;
      break;
    
    case DIAMS:
    case CLUBS:
      nPerTrick = 20;
      break;
    }
    
    nScore += nLevel * nPerTrick; // contract score
    if (nRisk > 0)
      nScore *= 2;  // doubled
    
    if (nRisk === 2)
      nScore *= 2;  // redoubled
    
    if (nScore < 100)
      nScore += 50; // part-score bonus
    else
      nScore += (bVul) ? 500 : 300; // game bonus
    
    if (nLevel === 6)
      nScore += (bVul) ? 750 : 500; // small slam bonus
    else if (nLevel === 7)
      nScore += (bVul) ? 1500 : 1000; // grand slam bonus
    
    nScore += 50 * nRisk; // for the insult (if any)
    
    var nPerOver = nPerTrick;
    if (nRisk) {
      nPerOver = (bVul) ? 200 : 100;
      if (nRisk > 1)
        nPerOver *= 2;
    }
    
    nScore += nOver * nPerOver; // overtricks
  }
  
  return nScore;
}

function GetPar(b) {
  /* The optimum score can be given in 4 possible formats: 
    "EW <score>"              score of EW 
    "NS <score>"              score of NS 
    "EW <score> NS <score>"   score of EW resp. NS 
    "NS <score> EW <score>"   score of NS resp. EW 
     where <score> is the integer number of points.
  */
  nspar = ewpar = 0;
  var t = b.TagValue('OptimumScore');
  var a = t.split(' ');
  if (a.length < 2) return false;
  if (a[0] === 'NS') {
    nspar = parseInt(a[1]);
    ewpar = -nspar;
  } else if (a[0] === 'EW') {
    ewpar = parseInt(a[1]);
    nspar = -ewpar;
  }
  if (a.length < 4) return true;
  if (a[2] === 'NS') {
    nspar = parseInt(a[3]);
  } else if (a[2] === 'EW') {
    ewpar = parseInt(a[3]);
  }
  return true;
}

var implimit = [10, 40, 80, 120, 160, 210, 260, 310, 360, 420, 490, 590,
740, 890, 1090, 1290, 1490, 1740, 1990, 2240, 2490, 2990, 3490, 3990];
function IMPS(score) {
  var imps;
  var a = Math.abs(score);
  if (a >= 4000) imps = 24;
  else {
    for (var ix in implimit) {
      if (a <= implimit[ix]) {
        imps = +ix;
        break;
      }
    }
  }
  if (score < 0) imps = -imps;
  return imps;
}

//  Create the CSV output file

var strPathname = bc.BrowseForFile('TabulateScores Output File', 1,
  'CSV Files (*.csv)|*.csv||');
if (!strPathname)
  WScript.Quit();

var fso = WScript.CreateObject('Scripting.FileSystemObject');
var fout = fso.CreateTextFile(strPathname, true);
fout.WriteLine('Deal,Contract,Vuln,By,Result,"Score NS","Score EW",'
  + '"Par Score","IMPS NS","IMPS EW","IMPS"');

//  Ensure double dummy analysis is complete and correct

 bc.DoubleDummyAllBoards();

var cBoards = bc.Boards.Count;
for (var iBoard = 0; iBoard < cBoards; ++iBoard) {
  var strLine = '';
  var brd = bc.Boards.Item(iBoard);
  
  //  Board
  
  var strBoard = brd.UniqueBoard;
  strLine += strBoard;
  
  var strVul = '';
  var strDeclarer = '';
  var strMake = '';
  var strScoreNS = '';
  var strScoreEW = '';
  var nPar = '';
  var strImpsNS = '';
  var strImpsEW = '';
  var strImps = '';
  
  //  Contract
  
  var strContract = brd.TagValue('Contract');
  
  //  Result
  
  var strResult = brd.TagValue('Result');
  
  //  Per the PBN standard, a pass-out should have
  //  Contract=="Pass" and Result=="", but we have seen
  //  Contract="" and Result="0".
  //  So we interpret both the above as a pass-out, otherwise
  //  we interpret Result="" as a no-play.
  
  if (strContract === '' && strResult === '0') {
    strContract = 'Pass';
    strResult = '';
  }
  
  if (strContract !== 'Pass' && strResult === '') {
    strContract = '';   // unfinished board, treat as no-play
  }
  
  if (strContract === 'Pass' || strResult !== '') {
    var nLevel = 0;
    var nStrain = NOTRUMP;
    var nRisk = 0;
    if (strContract !== 'Pass') {
      nLevel = parseInt(strContract.charAt(0));
      nStrain = 'CDHSN'.indexOf(strContract.charAt(1));
      for (var i = 2; i < strContract.length; ++i) {
        var ch = strContract.charAt(i);
        if (ch === 'X')
          ++nRisk;
      }
    }
    
    var nResult = parseInt(strResult);
    if (nLevel > 0) {
      strMake = nResult - (nLevel + 6);
      if (strMake === 0)
        strMake = '"= "';
    } else {
      strMake = nResult;
      if (strMake === 0)
        strMake = '';
    }
        
    //  Declarer
    
    strDeclarer = brd.TagValue('Declarer');
    var bNS = (strDeclarer === 'N' || strDeclarer === 'S');
    
    //  Vulnerability
      
    strVul = brd.TagValue('Vulnerable');
    var bVul = false;
    switch (strVul) {
    case 'None':
      break;
      
    case 'NS':
      if (bNS)
        bVul = true;
    break;
    
    case 'EW':
      if (!bNS)
        bVul = true;
      break;
      
    case 'All':
      bVul = true;
      break;
    }
    
    if (strVul === 'All')
      strVul = 'Both';
    
    //  Score
    
    var nScore = CalcScore(nLevel, nStrain, nRisk, bVul, nResult);      
    if (!bNS)
      nScore = -nScore;
    
    if (nScore > 0)
        strScoreNS = nScore;
    else if (nScore < 0)
        strScoreEW = -nScore;
  
    //  Par score
    
    var nspar = 0;
    var ewpar = 0;
    GetPar(brd);
    nPar = (bNS) ? nspar : -ewpar;
    
    //  Par IMPs
    
    var nImps = IMPS(nScore - nPar);
    var nAbs = Math.abs(nImps);
    
    if (nImps > 0) {
      strImpsNS = nAbs;
      strImpsEW = -nAbs;
      strImps = -nAbs;
    } else if (nImps < 0) {
      strImpsNS = -nAbs;
      strImpsEW = nAbs;
      strImps = -nAbs;
    } else {
      strImpsNS = 0;
      strImpsEW = 0;
      strImps = 0;
    }
  }
  
  strLine += ',' + strContract;
  strLine += ',' + strVul;
  strLine += ',' + strDeclarer;
  strLine += ',' + strMake;
  strLine += ',' + strScoreNS;
  strLine += ',' + strScoreEW;
  strLine += ',' + nPar;
  strLine += ',' + strImpsNS;
  strLine += ',' + strImpsEW;
  strLine += ',' + strImps;
  
  fout.WriteLine(strLine);
}
