// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com

//  This script extracts hand record data from all the boards in a bridge document,
//  and outputs the data to a spreadsheet CSV (comma-separated values) file.
//
//  Specify the optional "c" argument to count specific boards.
//  Modify the "count selection" expression, below, to select the boards to be counted.

//  $Id: TabulateBoards.js 182 2023-12-12 15:40:40Z Ray $

var bc = WScript.CreateObject('BridgeComposer.Object');
if (WScript.Arguments.length > 0) {
  bc.Open(WScript.Arguments(0));
} else {
  if (!bc.Open()) WScript.Quit();
}

var arg_c = WScript.Arguments.length > 1 && WScript.Arguments(1).indexOf('c') >= 0;

var CLUBS = 0;
var DIAMS = 1;
var HEARTS = 2;
var SPADES = 3;
var NOTRUMP = 4;
var NSTRAINS = 5;

function CalcScore(nLevel, nStrain, nRisk, bVul, nResult)
{
  //  nLevel = bid level (1..7) or 0 for pass-out
  //  nStrain = bid suit or notrump (see above)
  //  nRisk = 0 (undoubled), 1 (doubled), or 2 (redoubled)
  //  bVul = "true" if declarer is vulnerable
  //  nResult = tricks taken by declarer (0..13)
  
  if (nLevel === 0)
    return 0;
  
  var nScore = 0;
  if (nResult < nLevel + 6) {
    var nDown = nLevel + 6 - nResult;
    nScore = (bVul) ? -100 : -50;
    if (nRisk === 0) {
      nScore *= nDown;
    } else {
      // doubled (or redoubled)
      nScore *= 2;
      if (bVul) {
        if (nDown > 1)
          nScore -= 300 * (nDown - 1);
      } else {
        if (nDown > 1)
          nScore -= 200 * (nDown - 1);
        
        if (nDown > 3)
          nScore -= 100 * (nDown - 3);
      }
      
      if (nRisk === 2)
        nScore *= 2;  // redoubled
    }
  } else {
    var nOver = nResult - 6 - nLevel;
    var nPerTrick;
    switch (nStrain) {
    case NOTRUMP:
      nScore = 10;
      // FALL-THRU
    case SPADES:
    case HEARTS:
      nPerTrick = 30;
      break;
    
    case DIAMS:
    case CLUBS:
      nPerTrick = 20;
      break;
    }
    
    nScore += nLevel * nPerTrick; // contract score
    if (nRisk > 0)
      nScore *= 2;  // doubled
    
    if (nRisk === 2)
      nScore *= 2;  // redoubled
    
    if (nScore < 100)
      nScore += 50; // part-score bonus
    else
      nScore += (bVul) ? 500 : 300; // game bonus
    
    if (nLevel === 6)
      nScore += (bVul) ? 750 : 500; // small slam bonus
    else if (nLevel === 7)
      nScore += (bVul) ? 1500 : 1000; // grand slam bonus
    
    nScore += 50 * nRisk; // for the insult (if any)
    
    var nPerOver = nPerTrick;
    if (nRisk) {
      nPerOver = (bVul) ? 200 : 100;
      if (nRisk > 1)
        nPerOver *= 2;
    }
    
    nScore += nOver * nPerOver; // overtricks
  }
  
  return nScore;
}

function DDTricks(strddt, iPlayer, iStrain)
{
  //  For 20-char ddt: iPlayer: 0=N, 1=S, 2=E, 3=W
  //  For 10-char ddt: iPlayer: 0=1st, 1=2nd

  var iCol = NOTRUMP - iStrain + iPlayer * NSTRAINS;
  var strTricks = strddt.charAt(iCol);
  var nTricks = parseInt(strTricks, 16);
  return nTricks;
}

function DDMax(strddt, strPlayers, strStrains)
{
  strPlayers = strPlayers.toLowerCase();
  strStrains = strStrains.toLowerCase();
  var nResult = 0;
  for (ip = 0; ip < strPlayers.length; ++ip) {
    var chPlayer = strPlayers.charAt(ip);
    var iPlayer = 'nsew'.indexOf(chPlayer);
    for (is = 0; is < strStrains.length; ++is) {
      var chStrain = strStrains.charAt(is);
      var iStrain = 'cdhsn'.indexOf(chStrain);
      
      var nt = DDTricks(strddt, iPlayer, iStrain);
      if (nResult < nt)
        nResult = nt;
    }
  }

  return nResult;
}

function CalcMakingScore(strddt, bVul, iPlayer, iStrain)
{
  //  N.B. Returns zero if player cannot make at least 7 tricks in the strain

  var nTricks = DDTricks(strddt, iPlayer, iStrain);
  var nScore = 0;
  if (nTricks > 6)
    nScore = CalcScore(nTricks - 6, iStrain, 0, bVul, nTricks);

  return nScore;
}

var maxContract;
function CalcHighScore(brd)
{
  maxContract = '';
  var strParContract = brd.TagValue('ParContract');
  if (strParContract.indexOf('X') < 0)
    return '';

  var ch = strParContract.charAt(0);
  var iSacSide = (ch == 'E' || ch == 'W');  // the sacrificing side: NS=0, EW=1
  var iSide = 1 - iSacSide;   // the outbid side
  var strSide = (iSide === 0) ? 'NS' : 'EW';

  var strVul = brd.TagValue('Vulnerable');
  var bVul = false;
  if (iSide == 0 && strVul == 'NS')
    bVul = true;
  else if (iSide == 1 && strVul == 'EW')
    bVul = true;
  else if (strVul == 'All')
    bVul = true;

  var strDDT = brd.TagValue('DoubleDummyTricks');
  var strddt = (iSide === 0) ? strDDT.substring(0, 10) : strDDT.substring(10, 20);

  var maxScore = 0;
  for (iPlayer = 0; iPlayer < 2; ++iPlayer) {
    for (iStrain = CLUBS; iStrain <= NOTRUMP; ++iStrain) {
      var nScore = CalcMakingScore(strddt, bVul, iPlayer, iStrain);
      if (nScore > maxScore) {
        maxScore = nScore;
      }
    }
  }

  for (iStrain = NOTRUMP; iStrain >= CLUBS; --iStrain) {
    var strPlayer = '';
    var maxPlayer;
    for (iPlayer = 0; iPlayer < 2; ++iPlayer) {
      var nScore = CalcMakingScore(strddt, bVul, iPlayer, iStrain);
      if (nScore === maxScore) {
        strPlayer += strSide.charAt(iPlayer);
        maxPlayer = iPlayer;
      }
    }

    if (strPlayer)
    {
      var nTricks = DDTricks(strddt, maxPlayer, iStrain);

      if (maxContract)
        maxContract += '; ';

      maxContract += strPlayer + ' ' + (nTricks - 6) + 'CDHSN'.charAt(iStrain);
    }
  }

  return strSide + ' ' + maxScore;
}


(function() {

  //  Create the CSV output file

  var strPathname = bc.BrowseForFile('TabulateBoards Output File', 1, 'CSV Files (*.csv)|*.csv||');
  if (!strPathname)
    WScript.Quit();

  var fso = WScript.CreateObject('Scripting.FileSystemObject');
  var fout = fso.CreateTextFile(strPathname, true);
  var strHdr = 'Board,Dealer,Vuln,Deal,DDTricks,OptimumScore,ParContract,HighScore,HighScoreContract';
  if (arg_c)
    strHdr += ',Count';

  fout.WriteLine(strHdr);
  bc.DoubleDummyAllBoards();  //  Ensure double dummy analysis is complete and correct

  var nTotal = 0;
  var cBoards = bc.Boards.Count;
  for (var iBoard = 0; iBoard < cBoards; ++iBoard) {
    var strLine = '';
    var brd = bc.Boards.Item(iBoard);
    
    //  Board
    
    var strBoard = brd.UniqueBoard;
    strLine += strBoard;
        
    //  Dealer

    var strDealer = brd.TagValue('Dealer');
      
    //  Vulnerability
      
    var strVul = brd.TagValue('Vulnerable');

    //  Deal

    var strDeal = brd.TagValue('Deal');
    
    //  Double Dummy Tricks

    var strDDT = brd.TagValue('DoubleDummyTricks');
    var strddt = '0x' + strDDT;

    //  Optimum Score

    var strPar = brd.TagValue('OptimumScore');

    //  Par Contract

    var strParContract = brd.TagValue('ParContract');

    //  HighScore, HighScoreContract

    var strHighScore = CalcHighScore(brd);

    //  Count (if "c" argument specified)

    var bCount = 0;
    if (arg_c) {

      //  "Count selection" for the "c" argument:
      //  Modify this expression to count the board or not, as you prefer:

      bCount =
        DDMax(strDDT, 'ns', 'h') === 10       // NS making exactly 4 hearts
        && DDMax(strDDT, 'ew', 's') >= 10;    // and EW making 4 (or more) spades
      
    }

    nTotal += bCount;
    
    strLine += ',' + strDealer;
    strLine += ',' + strVul;
    strLine += ',' + strDeal;
    strLine += ',' + strddt;
    strLine += ',' + strPar;
    strLine += ',' + strParContract;
    strLine += ',' + strHighScore;
    strLine += ',' + maxContract;

    if (arg_c)
      strLine += ',' + ((bCount) ? '1' : '0');
    
    fout.WriteLine(strLine);
  }

  if (arg_c)
    bc.alert('Total count: ' + nTotal, 64);;
}());
