// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com

//  This script creates image files (type PNG) for all boards
//  in a document file.
//  You will be prompted for a folder in which to save the image files.
//  (Create and use a folder dedicated to this purpose, because all existing
//  files and subfolders will be deleted.)
//  The image files are named beginning with a sequence number so that
//  sorting by name will ensure they are in the correct order.
//
//  You may use this script as part of a process to create slideshows
//  and bridge movies. For example, in BridgeComposer:
//    Tools>All Boards
//    Edit>Select All
//    Board>Adjust View, clear Play checkbox (optional)
//    Board>Create Slides
//    click Close
//    Script>Slideshow
//
//    You may view the images as a slide show by using, for example,
//    the Windows Photos app.
//    (You may need to make registry changes for Photos on Windows 11.
//    Search the web for more info).
//
//    You may create a slideshow movie (type MP4, for example) using
//    Microsoft Clipchamp, OBS Studio, or similar.

//  $Id: Slideshow.js 247 2025-10-31 15:00:08Z Ray $


var MB_ICONWARNING = 48;
var MB_DEFBUTTON2 = 256;
var BCP_UPDATE = 0;
var BCP_TITLE = 1;
var BCP_FINISH = 4;
var IMAGE_PNG = 1;

//  created images parameters: adjust as desired

var IMAGE_PPI = 300;      // pixels per inch
var IMAGE_MARGIN = 0.25;  // margins (inches)

var bc = WScript.CreateObject('BridgeComposer.Object');
var fso = WScript.CreateObject('Scripting.FileSystemObject');


function DeleteAll(v)
{
  var e = new Enumerator(v);
  for (; !e.atEnd(); e.moveNext()) {
    var item = e.item();
    item.Delete(true);
  }
}


function ClearFolder(pathFolder)
{
  var fldr = fso.GetFolder(pathFolder);
  var vF = fldr.Files;
  var vSF = fldr.SubFolders;
  if (vF.Count || vSF.Count) {
    var str = 'All existing ';
    if (vF.Count)
      str += 'files ';

    if (vSF.Count)
    {
      if (vF.Count)
        str += 'and ';

      str += 'subfolers ';
    }

    str += 'in ' + pathFolder + ' will be deleted.';
    str += '\n\nClick OK to continue.';
    var br = bc.confirm(str, MB_ICONWARNING | MB_DEFBUTTON2);
    if (!br)
      WScript.Quit();

    DeleteAll(vSF);
    DeleteAll(vF);
  }
}


(function () {
  if (WScript.Arguments.length > 0 && WScript.Arguments.Item(0) !== '-') {
    bc.Open(WScript.Arguments(0));
  } else {
    if (!bc.Open())
      WScript.Quit();
  }

  var bds = bc.Boards;
  var nBoards = bds.Count;
  if (nBoards == 0)
  {
    bc.alert('Empty document');
    WScript.Quit();
  }

  var pathFolder = bc.BrowseForFolder('Destination Folder - Slideshow');
  if (!pathFolder)
    WScript.Quit();

  ClearFolder(pathFolder);

  bc.progress(BCP_TITLE, 'Slideshow');
  var iBoard = 0;
  while (bds.MoveNext())
  {
    var bd = bds.Current;
    ++iBoard;

  //  generate a unique name for the board
  //  avoid using invalid filename characters
  
    var strBoard = bd.UniqueBoard.replace(/[<>:"/\\|\?\*]/g, '#');
    var br = bc.progress(BCP_UPDATE, 'Creating ' + strBoard, iBoard / nBoards);
    if (!br)
      WScript.Quit();

    var strFile = iBoard.toString();
    while (strFile.length < 4)
      strFile = '0' + strFile;

    strFile = 'S' + strFile;
    strFile += ' ' + strBoard;
    var pathImg = pathFolder + '\\' + strFile + '.png';
    bd.WriteImageFile(pathImg, IMAGE_PNG, IMAGE_PPI, IMAGE_MARGIN, 0);
  }

  bc.progress(BCP_FINISH, 'Created ' + nBoards + ' images in folder ' + pathFolder);
})();
