// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com

//  This script allows you to show only selected boards in a hand record file
//  (and hide the others).
//
//  An optional argument may be specified:
//    -: Prompt for the argument value.
//    A single number: Shows that board number, hides all others.
//    A range (number-number): Shows board numbers from the first number
//      through the second number inclusive, hides all others.
//    *: Shows all boards.
//
//  If this optional argument is omitted, the user is prompted for it.
//
//  An additional optional argument may be specified, any combination of
//  the following letters:
//    p: Print the modified document (1 copy, as written).
//    f: Save the modified document as PDF (user prompted for PDF filename).
//    n: Do not Save the modified document.
//
//  Note: The deal set statistics (shown in hand record page layout
//  when enabled, and space permitting) will reflect only the currently
//  unhidden boards.
//
//  (This script may be placed in the BridgeComposer Script menu using the
//  Script>Favorites command. The optional arguments may be specified in
//  the Favorites dialog. A given script may appear multiple times in the
//  Favorites, each instance with different arguments.)
//
//  As written, this script shows and hides only the Diagram section.
//  This is adequate to show and hide boards in hand record page layout.
//  If you wish to use other page layouts, this script may require modification
//  to meet your needs.

//  $Id: ShowBoards.js 88 2022-04-17 15:41:03Z Ray $

var MB_ICONINFORMATION = 64;
var MB_ICONWARNING = 48;
var MB_ICONERROR = 16;
var BCF_SHOWDIAGRAM = 8;
var MB_YESNO = 4;
var MB_DEFBUTTON2 = 256;
var BIGNUM = 99999999;

var bc = WScript.CreateObject('BridgeComposer.Object');
if (WScript.Arguments.length > 0 && WScript.Arguments(0) !== '-') {
  bc.Open(WScript.Arguments(0));
} else {
  if (!bc.Open()) WScript.Quit();
}

var strRange = '';
var strPrompt = '1-99'
var nFirst, nLast;
if (WScript.Arguments.length > 1)
  strRange = WScript.Arguments(1);

for (;;) {
  if (!strRange || strRange === '-') {
    strRange = bc.prompt('Enter a range of board numbers to show, or *', strPrompt);
    if (strRange === null)
      WScript.Quit();
  }

  if (strRange === '*') {
    nFirst = 0;
    nLast = BIGNUM;
    break;
  }

  var vRange = strRange.split('-');
  if (vRange.length == 1) {
    nFirst = nLast = parseInt(vRange[0], 10);
  } else if (vRange.length == 2) {
    nFirst = parseInt(vRange[0], 10);
    if (vRange[1] === '*')
      nLast = BIGNUM;
    else
      nLast = parseInt(vRange[1], 10);
  }

  if (isFinite(nFirst) && isFinite(nLast) && nFirst >= 1 && nLast >= nFirst)
    break;

  bc.alert('Invalid range: ' + strRange, MB_ICONERROR);
  strPrompt = strRange;
  strRange = '';
}

var bds = bc.Boards;
var cBoards = bds.Count;
var iSelect = -1;
var nShowing = 0;
for (var iBoard = 0; iBoard < cBoards; ++iBoard)
{
  var bd = bds.Item(iBoard);
  var nBoard = parseInt(bd.TagValue('Board'), 10);
  var nFlags = parseInt(bd.TagValue('BCFlags'), 16);
  if (nBoard >= nFirst && nBoard <= nLast) {
    nFlags |= BCF_SHOWDIAGRAM;
    ++nShowing;
    if (iSelect < 0)
      iSelect = iBoard;
  }
  else
    nFlags &= ~BCF_SHOWDIAGRAM;

  bd.TagValue('BCFlags') = nFlags.toString(16);
}

var bPrint = false;
var bPDF = false;
var bNoSave = false;
if (WScript.Arguments.length > 2) {
  var strOpt = WScript.Arguments(2);
  bPrint = strOpt.indexOf('p') >= 0;
  bPDF = strOpt.indexOf('f') >= 0;
  bNoSave = strOpt.indexOf('n') >= 0;
}

if (iSelect >= 0)
  bc.SelectedBoard = iSelect;

if (bPrint) {
  if (!nShowing) {
    bPrint = bc.confirm('No boards are showing. Print anyway?', MB_ICONINFORMATION | MB_YESNO | MB_DEFBUTTON2);
  }

  if (bPrint)
    bc.Print(1, 0);
}

if (bPDF) {
  if (!nShowing) {
    bPDF = bc.confirm('No boards are showing. Save as PDF anyway?', MB_ICONINFORMATION | MB_YESNO | MB_DEFBUTTON2);
  }

  if (bPDF) {
    var file = bc.CallerPathname;
    if (!file)
      file = bc.pathname;

    var ix = file.lastIndexOf('\\');
    if (ix >= 0)
      file = file.substr(ix + 1);

    ix = file.lastIndexOf('.');
    if (ix >= 0)
      file = file.substr(0, ix) + '.pdf';

    var path = bc.BrowseForFile('Save As PDF - BridgeComposer', 1, 'PDF files (*.pdf)|*.pdf||', file)
    if (path)
      bc.SaveAsPdf(path);
  }
}

if (!bNoSave)
  bc.Save();
