// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com
//
//  This script sets the Dealer for all boards in a document.
//
//  The Deal is rotated so that the new Dealer will get the North hand in
//  the original the Deal tag. (In this script, the original dealer is ignored.)
//
//  If the boards are not already numbered, they are numbered starting with 1 (one).
//
//  You are prompted to "Enter Dealer pattern". The pattern gives the sequence
//  of players to be set as Dealer. When the pattern is used up, it repeats from the
//  beginning. Examples:
//    n       Set Dealer to North in all boards.
//    ns      Sets the first board Dealer to North, second to South, and repeats.
//    nesw    Sets the normal Dealer sequence, that is, each player in turn, clockwise.
//
//  The resulting document is automatically exported as a LIN file.
//
//  Note that if a LIN file with the same name in the same folder already exists,
//  it is overwritten without warning.
//
//  $Id: SetDealer.js 223 2024-12-20 19:31:50Z Ray $


var NHANDS = 4;
var strPlayer = 'NESW';
var bc = WScript.CreateObject('BridgeComposer.Object');
if (WScript.Arguments.length > 0 && WScript.Arguments(0) !== '-') {
  bc.Open(WScript.Arguments(0));
} else {
  if (!bc.Open()) WScript.Quit();
}

var strOption = '';
if (WScript.Arguments.length > 1) {
  strOption = WScript.Arguments(1);
}

while (!/^[NnEeSsWw]+$/.test(strOption)) {
  if (strOption === '') strOption = 'n';
  strOption = bc.prompt('Enter Dealer pattern', strOption);
  if (strOption === null) WScript.Quit();
}

strOption = strOption.toUpperCase();
var ixDealer = 0;
var bds = bc.Boards;
var cBoards = bds.Count;
for (var iBoard = 0; iBoard < cBoards; ++iBoard) {
  var bd = bds.Item(iBoard);
  var strDeal = bd.TagValue('Deal').toUpperCase();
  var vDeal = strDeal.substr(2).split(' ');
  var chOrig = strDeal.charAt(0);
  var nOrig = strPlayer.indexOf(chOrig);
  var ixHand = (-nOrig) & 3;   // index of North hand
  var chDealer = strOption.charAt(ixDealer);
  var nDealer = strPlayer.indexOf(chDealer);
  var strNewdeal = chDealer + ':';
  for (var i = 0; i < NHANDS; ++i) {
    strNewdeal += vDeal[ixHand];
    if (i < NHANDS - 1)
      strNewdeal += ' ';

    ixHand = (ixHand + 1) & 3;
  }
  
  bd.TagValue('Deal') = strNewdeal;
  bd.TagValue('Dealer') = chDealer;

  if (nDealer & 1) {
    var strVul = bd.TagValue('Vulnerable');
    if (strVul === 'NS')
      bd.TagValue('Vulnerable') = 'EW';
    else if (strVul === 'EW')
      bd.TagValue('Vulnerable') = 'NS';
  }

  if (bd.TagValue('Board') === '')
    bd.TagValue('Board') = iBoard + 1;

  ++ixDealer;
  if (ixDealer >= strOption.length)
    ixDealer = 0;
}

bc.Save();
var bAlert = false;
var strPath = bc.CallerPathname;
if (!strPath) {
  strPath = bc.pathname;
  bAlert = true;
}

var ix = strPath.lastIndexOf('.');
if (ix >= 0)
  strPath = strPath.substr(0, ix);

try {
  strPath += '.lin';
  bc.ExportAsBridgeBaseOnline(strPath);
  if (bAlert)
    bc.alert('Exported ' + strPath, 64);
} catch(ex) {}
