// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com

//  This script checks a PBN file to verify that it is a correct
//  duplicate bridge event board set. Specifically, it checks that:
//  - the first board number is 1, and subsequent board numbers
//    increase by one
//  - the dealer and vulnerable fields are the correct standard values
//    based on the board number.
//
//  At completion, a dialog panel is displayed listing deviations from
//  the expected values, if any, or saying "No problems found".
//
//  The exit code is 0 if no problems were found, else 1.
//
//  This script does not modify the input PBN file.

//  $Id: SetCheck.js 222 2024-12-15 21:13:01Z Ray $


//  Message box severity

var MB_ICONINFORMATION = 64;
var MB_ICONWARNING = 48
var MB_ICONERROR = 16

//  BridgeComposer object

var bc = WScript.CreateObject('BridgeComposer.Object');
if (WScript.Arguments.length > 0 && WScript.Arguments(0) !== '-') {
  bc.Open(WScript.Arguments(0));
} else {
  if (!bc.Open()) WScript.Quit();
}

//  Diagnostic reporting

var strReport = '';
function Diag(str)
{
  strReport += str + '\n';
}


(function()
{
  var vVul = ['None', 'NS', 'EW', 'All'];
  var nBoardExpected = 1;
  var cBoards = 0;

  var bds = bc.Boards;
  while (bds.MoveNext()) {
    cBoards++;
    var bd = bds.Current;
    var strBoard = bd.TagValue('Board');

    //  check board number valid

    var nBoard = parseInt(strBoard);
    var vm = strBoard.match(/^\d+$/);
    if (nBoard < 1 || vm === null) {
      Diag('Expecting Board ' + nBoardExpected + ', found "' + strBoard + '"');
      continue;
    }

    //  check board number is expected

    if (nBoard !== nBoardExpected) {
      Diag('Expecting Board ' + nBoardExpected + ', found Board ' + strBoard);
    }

    //  check Dealer

    var iDealer = (nBoard - 1) & 3;
    var chDealer = 'NESW'.charAt(iDealer);
    var strDealer = bd.TagValue('Dealer');
    if (strDealer !== chDealer)
    {
      Diag('Board ' + strBoard + ': Incorrect Dealer: Expected ' + chDealer + ', found ' + strDealer);
    }

    //  check Vulnerable

    var iBoard = nBoard - 1;
    var iVul = ((iBoard >> 2) + iBoard) & 3;
    var strExpect = vVul[iVul];
    var strVul = bd.TagValue('Vulnerable');
    if (strVul !== strExpect) {
      Diag('Board ' + strBoard + ': Incorrect Vulnerable: Expected ' + strExpect + ', found ' + strVul);
    }

    nBoardExpected = nBoard + 1;
  }

  var nExit = 0;
  if (!strReport) {
    var strPlural = (cBoards === 1) ? '' : 's';
    strReport = 'SetCheck:\n' + cBoards + ' board' + strPlural + ' examined\nNo problems found';
    var nType = MB_ICONINFORMATION;
  } else {
    var nType = MB_ICONWARNING;
    nExit = 1;
  }

  bc.alert(strReport, nType);
  WScript.Quit(nExit);
})();
