// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com
//
//  This script rotates all boards in a document, as necessary, so that
//  a specific player position is the declarer (South by default).
//
//  To rotate the declarer to a position other than South, specify a command line argument
//  of "n", "e", or "w" for North, East, or West, respectively.
//
//  Boards that do not have the Declarer tag set, or that have the Declarer tag already
//  set to the desired player position, are not modified.
//
//  The rotated PBN tags are Dealer, Vulnerable, Deal, Auction, Declarer, Play, and Score;
//  commentary text is also rotated.
//
//  $Id: RotateDeclarer.js 177 2023-12-02 20:55:24Z Ray $


var bc = WScript.CreateObject('BridgeComposer.Object');
var strPlayer = 'NESW';


function RotateCommentary(bd, nRot)
{
  //  Rotates the words "North", "East", "South", and "West" in commentary

  for (var nCmty = 0; nCmty < 4; ++nCmty)
  {
    // Based on https://stackoverflow.com/questions/14013223/how-to-replace-multiple-keywords-by-corresponding-keywords-with-the-replace-me

    var strOld = bd.Commentary(nCmty);
    var strNew = strOld.replace(/\b(north|east|south|west)\b/gi,
      function(mat)
      {
        var ch = mat.charAt(0).toLowerCase();
        var ix = 'nesw'.indexOf(ch);
        var repl = ['north', 'east', 'south', 'west'][(ix + nRot) & 3];
        if (/^[A-Z][a-z]/.test(mat)) // initial cap
          repl = repl.substr(0, 1).toUpperCase() + repl.substr(1);
        else if (/^[A-Z][A-Z]/.test(mat)) // all caps
          repl = repl.toUpperCase ();
        return repl;
      });

    bd.Commentary(nCmty) = strNew;
  }
}


function RotatePlayerChar(ch, nRot)
{
  //  Rotates the chars "N", "E", "S", and "W". Returns the rotated char.

  if (ch.length === 1) {
    var n = strPlayer.indexOf(ch);
    n = (n + nRot) & 3;
    ch = strPlayer.charAt(n);
  }
  return ch;
}


function RotatePlayerTag(bd, strTag, nRot)
{
  //  For a PBN tag, rotates a one-char player position value.
  //  Returns the rotated char, if rotated, else the original value.

  var ch = bd.TagValue(strTag);
  if (ch.length === 1) {
    ch = RotatePlayerChar(ch, nRot);
    bd.TagValue(strTag) = ch;
  }
  return ch;
}


function RotateSideTag(bd, strTag)
{
  //  For a PBN tag, changes its value starting with "NS" to "EW", and vice-versa.
  //  Additional chars in the value (if any) are not modified.

  var str = bd.TagValue(strTag);
  var strSide = str.substr(0, 2);
  if (strSide === 'NS')
    strSide = 'EW';
  else if (strSide === 'EW')
    strSide = 'NS';
  else
    return;

  str = strSide + str.substr(2);
  bd.TagValue(strTag) = str;
}


function RotateBoard(bd, nRot)
{
  //  Rotates the board "bd" clockwise by "nRot" positions

  var chDealer = RotatePlayerTag(bd, 'Dealer', nRot);

  var strDeal = bd.TagValue('Deal').toUpperCase();
  var strNewdeal = chDealer + strDeal.substr(1);	  
  bd.TagValue('Deal') = strNewdeal;

  RotatePlayerTag(bd, 'Auction', nRot);
  RotatePlayerTag(bd, 'Declarer', nRot);
  RotatePlayerTag(bd, 'Play', nRot);

  if ((nRot & 1) != 0) {
    // rotating an odd (not even) number of positions
    RotateSideTag(bd, 'Vulnerable');
    RotateSideTag(bd, 'Score');
  }

  RotateCommentary(bd, nRot);
}


(function()
{
  if (WScript.Arguments.length > 0 && WScript.Arguments(0) !== '-') {
    bc.Open(WScript.Arguments(0));
  } else {
    if (!bc.Open()) WScript.Quit();
  }

  var chDest = 'S';
  if (WScript.Arguments.length > 1) {
    var str = WScript.Arguments(1);
    if (str.length === 1) {
      var ch = str.toUpperCase();
      if (strPlayer.indexOf(ch) >= 0)
        chDest = ch;
    }
  }

  var nDest = strPlayer.indexOf(chDest);
  var nMod = 0;
  var bds = bc.Boards;
  while (bds.MoveNext())
  {
    var bd = bds.Current;
    var chDeclarer = bd.TagValue('Declarer');
    if (chDeclarer.length !== 1)
      continue;

    var nDeclarer = strPlayer.indexOf(chDeclarer);
    if (nDeclarer < 0)
      continue;

    var nRot = (nDest - nDeclarer) & 3;
    if (nRot === 0)
      continue;

    RotateBoard(bd, nRot);
    ++nMod;
  }

  if (nMod)
    bc.Save();
})();
