// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com

//  Some older programs do not properly handle standard PBN files.
//  This script modifies the PBN in an attempt to be more compatible with such programs.
//  Currently, all the changes are to the Auction section:
//    1. Auction call suffixes are changed into Notes, so no suffixes or NAGs are used.
//    2. PBN comments in the Auction section are removed.
//    3. The Auction AP (All pass) token is changed into 3 (or 4) Pass tokens.
//    4. Empty text in Auction Notes is changed to "--".

//  $Id: Retroize.js 58 2021-08-09 15:35:24Z Ray $


var bc = WScript.CreateObject('BridgeComposer.Object');
var fso = WScript.CreateObject('Scripting.FileSystemObject');


function stricmp(str1, str2)
{
  //  Compares two strings without regard to case,
  //  and returns zero if equivalent.
  //  Uppercase is used rather than lowercase in this function,
  //  due to issues with certain UTF-8 character conversions.
  
  return str1.toUpperCase().localeCompare(str2.toUpperCase());
}


function isDigit(ch)
{
  return /^[0-9]$/.test(ch);
}


function isCallCardChar(ch)
{
  return /^[A-Za-z0-9\-]$/.test(ch);
}


function isBid(str)
{
  //  standard relaxed to accept N as an alternative for NT
  return /^[1-7](S|H|D|C|N|NT)$/.test(str.toUpperCase());
}


var ttNone = 0;
var ttError = 1;
var ttNoMore = 2;
var ttSelfDelim = 3;
var ttCommentLine = 4;
var ttCommentBrace = 5;
var ttString = 6;
var ttSuffix = 7;
var ttNote = 8;
var ttNAG = 9;
var ttIrregular = 10;
var ttCallCard = 11;


function NextToken(section)
{
  //  Get the next token from the Auction section
  //  Lexical analysis as per the PBN standard version 2.1
  
  section.strToken = "";
  var ixLimit = section.text.length;      
  var tt = ttNone;
  section.nTokenType = tt;
  section.ixStart = section.ix;
  var ixFinish = section.ixStart;
  var bExpect = false;    // another char is expected
  for (;;) {
    if (section.ix >= ixLimit) {
      if (bExpect) {
        tt = ttError;
        --section.ixStart;
      }
      
      section.nTokenType = (tt === ttNone) ? ttNoMore : tt;
    }
    
    if (section.nTokenType !== ttNone) {
      section.strToken = section.text.substring(section.ixStart, ixFinish);
      return;
    }
    
    var ch = section.text.charAt(section.ix++);
    ixFinish = section.ix;
    switch (tt) {
    case ttNone:        // classify by first char of new token
      switch (ch) {
      case '\r':
      case '\n':
      case '\t':
      case ' ':
        section.ixStart = section.ix;
        break;    // skip delimiter char
     
      case '[':
      case ']':
      case '*':
      case '+':
        section.strToken = ch;
        section.nTokenType = ttSelfDelim;
        return;
      
      case ';':
        tt = ttCommentLine;
        section.ixStart = section.ix;
        break;
      
      case '{':
        tt = ttCommentBrace;
        section.ixStart = section.ix;
        bExpect = true;
        break;
        
      case '"':
        tt = ttString;
        section.ixStart = section.ix;
        bExpect = true;
        break;
        
      case '!':
      case '?':
        tt = ttSuffix;
        break;
        
      case '=':
        tt = ttNote;
        section.ixStart = section.ix;
        bExpect = true;
        break;
        
      case '$':
        tt = ttNAG;
        section.ixStart = section.ix;
        break;
        
      case '^':
        tt = ttIrregular;
        section.ixStart = section.ix;
        bExpect = true;
        break;
        
      default:
        if (isCallCardChar(ch)) {
          tt = ttCallCard;
          break;
        } else {
          section.strToken = ch;
          section.nTokenType = ttError;
          return;
        }
      }
      break;
      
    //  determine when new token has ended
      
    case ttCommentLine:
      if (ch === '\r' || ch === '\n') {
        section.nTokenType = tt;
        --section.ix;
      }
      break;
    
    case ttCommentBrace:
      if (ch === '}') {
        section.nTokenType = tt;
        --ixFinish;
      }
      break;
      
    case ttString:
      if (ch === '"') {
        section.nTokenType = ttString;
        return;
      }
      
      if (ch === '\\') {
        if (section.ix >= ixLimit) {
          section.nTokenType = ttError;
          return;
        }
          
        ch = section.text.charAt(section.ix++);
      }
      
      section.strToken += ch;
      break;
      
    case ttSuffix:
      if ((ch !== '!' && ch !== '?') || ixFinish > section.ixStart + 2) {
        section.nTokenType = tt;
        --ixFinish;
        --section.ix;
      }
      break;
      
    case ttNote:
      if (ch === '=') {
        section.nTokenType = tt;
        --ixFinish;
      }
      break;
      
    case ttNAG:
      if (!isDigit(ch)) {
        section.nTokenType = tt;
        --ixFinish;
        --section.ix;
      }
      break;
      
    case ttIrregular:
      if ('ISRL'.indexOf(ch) < 0) {
        tt = ttError;
        --section.ixStart;
      }
      
      section.nTokenType = tt;
      bExpect = false;
      break;
      
    case ttCallCard:
      if (!isCallCardChar(ch)) {
        section.nTokenType = tt;
        --ixFinish;
        --section.ix;
      }
      break;
    }
  }
}

function ParseAuction(strAuction)
{
  var bBid = false;
  var section = {ix: 0, text: strAuction};
  var vToken = [];
  for (;;) {
    NextToken(section);
    if (section.nTokenType === ttNoMore)
      break;
    
    switch (section.nTokenType) {
    case ttCallCard:
      if (isBid(section.strToken))
        bBid = true;

      var tok = {token: section.strToken};
      vToken.push(tok);
      break;

    case ttNote:
      vToken[vToken.length - 1].note = +section.strToken;
      break;

    case ttSuffix:
      vToken[vToken.length - 1].suffix = section.strToken;
      break;
    }
  }

  //  Replace trailing AP with 3 or 4 Pass

  if (vToken.length >= 1) {
    var tok = vToken[vToken.length - 1];
    if (stricmp(tok.token, 'AP') === 0) {
      vToken.pop();
      var tokPass = {token: 'Pass'};
      var nPass = (bBid) ? 3 : 4;
      while (nPass--) {
        vToken.push(tokPass);
      }
    }
  }

  return vToken;
}


function AdjustNotes(vAuction, vNote)
{
  //  Turn suffixes into Notes

  var nNote = 0;
  for (var ix in vAuction) {
    if (vAuction[ix].note) {
      vAuction[ix].note = (nNote + 1);
      ++nNote;
    } else if (vAuction[ix].suffix) {
      var note = { number: (nNote + 1), text: vAuction[ix].suffix };
      vNote.splice(nNote, 0, note);
      vAuction[ix].note = ++nNote;
    }
  }
}


function ProduceAuction(vToken)
{
  var str = '';
  var nToken = 0;
  for (var ix in vToken) {
    var tok = vToken[ix];
    if (nToken >= 4) {
      str += '\n';
      nToken = 0;
    }

    if (nToken > 0)
      str += ' ';
      
    str += tok.token;

    if (tok.note) {
      str += ' =';
      str += tok.note;
      str += '=';
    }
    
    ++nToken;
  }

  str += '\n';
  return str;
}


(function()
{
  if (WScript.Arguments.length > 0 && WScript.Arguments(0) !== '-') {
    bc.Open(WScript.Arguments(0));
  } else {
    if (!bc.Open()) WScript.Quit();
  }

  var bds = bc.Boards;
  while (bds.MoveNext()) {
    var bd = bds.Current;
    var strAuction = bd.TagSection('Auction');
    var vAuction = ParseAuction(strAuction);

    var vNote = [];
    var notes = bd.AuctionNotes;
    while (notes.MoveNext()) {
      var strNote = notes.Current.value;
      var vMatch = strNote.match(/(\d+):(.*)/);
      if (vMatch && vMatch.length >= 3) {
        var objNote = { number: vMatch[1], text: vMatch[2] };
        if (objNote.text === '')
          objNote.text = '--';    // change empty to "--" (notation for void?)
        
        vNote.push(objNote);
      }
    }

    AdjustNotes(vAuction, vNote);
    var strAuctionNew = ProduceAuction(vAuction);
    bd.TagSection('Auction') = strAuctionNew;

    notes.RemoveAll();
    for (var ix in vNote) {
      var note = vNote[ix];
      var strNote = (+ix + 1) + ':' + note.text;
      notes.Add(strNote);
    }
  }

  bc.Save();
})();
