// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com

//  This is an experimental script to re-score pair games
//  using alternative scoring methods.

//  The boards must already have Score Tables, normally created
//  using BC "Tools>Import Score Data".
//  This script will add two new Score Table columns: IMP_NS and IMP_EW.
//  Only single-section, Mitchell movement games are currently supported.
//  Also, a Total Score Table will be inserted at the beginning
//  of the file to display each pair's total score.

//  Pairs that happen to play fewer boards than the others
//  will have their total score "factored". 
//  Note that at IMP and BAM scoring, this will make a negative score more negative;
//  but this seems necessary for fairness to the rest of the players
//  (pairs having a poor game should not benefit from sitting out).

//  You may adjust some Total Score Table options
//  (notably "order by pair number" vs. "order by rank")
//  using BC "Format>Board Layout", Score Table tab.

//  Settings will be loaded from "BCReloadLayout.pbn" in your
//  "Documents" folder. This file will be created if it does not exist.
//  You may open this file in BC and use "Format>Board Layout"
//  to adjust its settings (including score table sort options).

//  This script may be called from the BC Script menu, in which case the open
//  document will be updated (but not saved).
//  It be called from a command line with the path of a document, in which
//  case that document will be overwritten (not recommended).
//  Or, it may be called without arguments (e.g. by double-clicking), in which
//  case you will be prompted for the input document and the output will be saved
//  to the same filename but with the scoring method appended.

//  Requires BC 5.62 or later.
//  Revised 2018-06-11

var bc = WScript.CreateObject('BridgeComposer.Object');
var bArg = false;
var path;
if (WScript.Arguments.length > 0) {
  bArg = true;
  path = WScript.Arguments.Item(0);
}

function Score(text) {
  if (text === "-") return 0;
  if (text === 'PASS') return 0;
  return Number(text);
}

var implimit = [10, 40, 80, 120, 160, 210, 260, 310, 360, 420, 490, 590,
740, 890, 1090, 1290, 1490, 1740, 1990, 2240, 2490, 2990, 3490, 3990];
function IMPS(score) {
  var imps;
  var a = Math.abs(score);
  if (a >= 4000) imps = 24;
  else {
    for (var ix in implimit) {
      if (a <= implimit[ix]) {
        imps = ix;
        break;
      }
    }
  }
  if (score < 0) imps = -imps;
  return Number(imps);
}

function GetPar(b) {
  /* The optimum score can be given in 4 possible formats: 
    "EW <score>"              score of EW 
    "NS <score>"              score of NS 
    "EW <score> NS <score>"   score of EW resp. NS 
    "NS <score> EW <score>"   score of NS resp. EW 
     where <score> is the integer number of points.
  */
  nspar = ewpar = 0;
  var t = b.TagValue('OptimumScore');
  var a = t.split(' ');
  if (a.length < 2) return false;
  if (a[0] === 'NS') {
    nspar = parseInt(a[1]);
    ewpar = -nspar;
  } else if (a[0] === 'EW') {
    ewpar = parseInt(a[1]);
    nspar = -ewpar;
  }
  if (a.length < 4) return true;
  if (a[2] === 'NS') {
    nspar = parseInt(a[3]);
  } else if (a[2] === 'EW') {
    ewpar = parseInt(a[3]);
  }
  return true;
}

var arrNS = [];    // NS pair data
var arrEW = [];    // EW pair data
var maxBoards = 0;  // max boards played by a pair, for MP factoring

//  Open the document

if (bArg) {
  bc.Open(path);
} else {
  if (!bc.Open()) WScript.Quit();
  path = bc.pathname;
}

//  Prompt for the desired scoring method

var sPrompt = 'Select scoring method:\n' +
  '0: Insert "Total Score Table" for original matchpoints\n' +
  '1: Cross IMPs\n' +
  '2: IMPs using double-dummy par as datum\n' +
  '3: BAM vs. double-dummy par\n';

var arMethod = [
  {scoring:'MP2', filetype:'MP', decimal:2},
  {scoring:'IMP;Cross2', filetype:'ImpCross', decimal: 2},
  {scoring:'IMP;Par', filetype:'ImpPar', decimal: 0},
  {scoring:'BAM;Par', filetype:'BamPar', decimal: 0}
];
var strScoring;
var strFileType;
var nDecimal;
for (var iTry = 0;; ++iTry) {
  var nMethod = bc.prompt(sPrompt, '1');
  if (!nMethod) WScript.Quit();
  nMethod = Number(nMethod);
  var i = nMethod;
  if (i >= 0 && i < arMethod.length) {
    strScoring = arMethod[i].scoring;
    strFileType = arMethod[i].filetype;
    nDecimal = arMethod[i].decimal;
    break;
  } else {
    if (iTry === 0) sPrompt = 'Incorrect selection, try again.\n\n' + sPrompt;
  }
}  

//  Load the layout options (including table sort order)

var wsh = WScript.CreateObject('WScript.Shell');
var pathDocs = wsh.SpecialFolders('MyDocuments');
var pathLayout= pathDocs + '\\BCRescoreLayout.pbn';
var fso = WScript.CreateObject('Scripting.FileSystemObject');
if (!fso.FileExists(pathLayout)) {
  var ts = fso.CreateTextFile(pathLayout);
  ts.WriteLine('%BCOptions Center GutterH GutterV STBorder STShade STSortScore');
  ts.WriteLine('%BoardsPerPage 1');
  ts.WriteLine('%TSTReportOrder ByRank');
  ts.Close();
}
bc.LoadLayout(pathLayout);

//  Append scoring method to Event title

var strEvent = bc.Event;
if (strEvent !== '') strEvent += ': ';
strEvent += '@' + strFileType;

//  Rescore each board in turn

var nValidMP = 0;
var nValidPar = 0;
var bds = bc.Boards;
for (var iBoard = 0; iBoard < bc.Boards.Count; ++iBoard)
{
  var b = bds.Item(iBoard);
  var stt = b.TagValue('ScoreTable');
  if (!stt.length) continue;  // no score table for this board
  
  //  Parse the ScoreTable tag to get column names
  //  The "col" object will give the column number for each column name
  
  var hdg = stt.split(';');  // N.B. hdg.length is the number of columns
  var col = {};
  var colname = [];
  for (var i = 0; i < hdg.length; ++i) {
    var aname = hdg[i].match(/[^\\]*/);
    var name = aname[0];
    if (name === 'IMP_NS' || name === 'IMP_EW') {
      var board = b.TagValue('Board');
      bc.alert('Board ' + board + ' already has IMP scoring.\nCannot continue.', 16);
      WScript.Quit(1);
    }
    colname[i] = name;
    col[name] = i;
  }
  
  if (nMethod === 2 || nMethod === 3) {
    var nspar, ewpar;
    if (!GetPar(b)) continue;
    ++nValidPar;
  }
  
  //  Parse the ScoreTable section (data rows and columns)
  
  var sts = b.TagSection('ScoreTable');
  var tok = sts.match(/"[^"]*"|\S+/g);  // split space-delimited PBN tokens into an array
  var stsnew = '';
  for (var i = 0; i < tok.length; i += hdg.length) {
    
    //  Process next row (one NS and EW pair result)
    
    var iscore = Score(tok[i + col.Score_NS]) - Score(tok[i + col.Score_EW]);
    var totns = 0, totew = 0;
    switch (nMethod) {
    case 1:
      //  IMP Cross:
      //  Each opponent's score is subtracted from your score and
      //  converted to IMPs. The IMPs are then summed and divided
      //  by the number of opponents.
      var total = 0;
      for (var j = 0; j < tok.length; j += hdg.length) {
        if (j !== i) {
          var jscore =  Score(tok[j + col.Score_NS]) - Score(tok[j + col.Score_EW]);
          var diff = iscore - jscore;
          var imps = IMPS(diff);
          total += imps;
        }
      }
      var n = tok.length / hdg.length - 1;
      if (n) total /= n;
      totns = total;
      totew = -total;
      break;
    case 2:
      //  IMP Par (experimental):
      //  The double-dummy par score is subtracted from your score and
      //  converted to IMPs.
      totns = IMPS(iscore - nspar);
      totew = IMPS(-iscore - ewpar);  
      break;
    case 3:
      //  BAM vs. Par (experimental):
      //  You get 1, 0, or -1 matchpoints per board as your score is
      //  greater than, equal to, or less than the double-dummy par score.
      //  Accumulated here in the "IMP" score fields.
      totns = (iscore > nspar) ? 1 : (iscore < nspar) ? -1 : 0;
      totew = (-iscore > ewpar) ? 1 : (-iscore < ewpar) ? -1 : 0;
      break;
    case 0:
      //  Add Total Score Table for original matchpoint data
      var mpns = tok[i + col.MP_NS];
      var mpew = tok[i + col.MP_EW];
      if (mpns !== '-' && mpew !== '-') {
        totns = Number(mpns);
        totew = Number(mpew);
        ++nValidMP;
      }
      break;
    }
    
    //  Accumulate total score for NS and EW pairs
    
    var iPair = tok[i + col.PairId_NS];
    if (!arrNS[iPair]) {
      arrNS[iPair] = {score: 0, boards: 0};
    }
    arrNS[iPair].pair = iPair;
    arrNS[iPair].names = tok[i + col.Names_NS];
    arrNS[iPair].score += Number(totns.toFixed(2));
    ++arrNS[iPair].boards;
    if (arrNS[iPair].boards > maxBoards)
      maxBoards = arrNS[iPair].boards;
    
    iPair = tok[i + col.PairId_EW];
    if (!arrEW[iPair]) {
      arrEW[iPair] = {score: 0, boards: 0};
    }
    arrEW[iPair].pair = iPair;
    arrEW[iPair].names = tok[i + col.Names_EW];
    arrEW[iPair].score += Number(totew.toFixed(2));
    ++arrEW[iPair].boards;
    if (arrEW[iPair].boards > maxBoards)
      maxBoards = arrEW[iPair].boards;
    
    //  Copy the existing column data in the ScoreTable section row
    
    for (var j = 0; j < colname.length; ++j) {
      if (j > 0) stsnew += ' ';
      stsnew += tok[i + j];
    }
    
    if (nMethod !== 0) {
      //  Add data for the new IMP_NS and IMP_EW columns
      
      stsnew += ' ' + totns.toFixed(nDecimal) + ' ' + totew.toFixed(nDecimal);
      stsnew += '\r\n';
    }
  }
  
  if (nMethod !== 0) {
    //  Append new column names to the ScoreTable tag
    //  (BC will adjust column width as needed).
    //  Update all tags for this board.
    
    var sttnew = stt + ';IMP_NS\\5R;IMP_EW\\5R';
    b.TagValue('ScoreTable') = sttnew;
    b.TagSection('ScoreTable') = stsnew;
    b.TagValue('Scoring') = strScoring;
  }
  
  //  Update Event tag
  
  var et = b.TagValue('Event');
  if (et !== '')
    b.TagValue('Event') = strEvent;
}
if (nMethod === 0 && nValidMP === 0) {
  bc.alert('For method 0, need matchpoint scores.\n' +
    'Import an ACBLscore .txt report with Traveler Scores.', 16);
    WScript.Quit(1);
}
if (nMethod === 2 || nMethod === 3) {
  if (nValidPar === 0) {
    bc.alert('For methods 2 and 3, need double-dummy par for each board.\n' +
    'Supply hand records and use "Tools>Double Dummy All Boards".', 16);
    WScript.Quit(1);
  }
}

function CmpImps(a, b) {
  return parseFloat(b.score) - parseFloat(a.score);
}

var tst = '';
function TST(arr, dir)
{
  //  Generate TotalScoreTable rows for one direction (N-S or E-W)
  
  if (true) {    // adjust condition to enable/disable factoring
    
    // factor scores based on number of boards played
    
    for (var i = 0; i < arr.length; ++i) {
      if (arr[i] && arr[i].boards !== maxBoards) {
        arr[i].score = arr[i].score / arr[i].boards * maxBoards;
        nDecimal = 2;
      }
    }
  }
    
  arr.sort(CmpImps);
  var rank, sLast = -999999, ix = 0;
  for (var i = 0; i < arr.length; ++i) {
    if (arr[i]) {
      ++ix;
      
      //  Calculate Rank taking ties into account
      
      var score = arr[i].score;
      if (score !== sLast) {
        sLast = score;
        rank = ix;
        var er = rank;
        var jx = ix;
        for (var j = i + 1; j < arr.length; ++j) {
          if (arr[j]) {
            ++jx;
            if (arr[j].score === score) er = jx;
            else break;
          }
        }
        if (er > rank) rank += '/' + er;
      }
      
      //  Generate one row
      
      tst += rank + ' ' + arr[i].pair + ' ' + dir + ' ';
      tst += score.toFixed(nDecimal) + ' ';
      tst += arr[i].boards + ' ';
      tst += arr[i].names + '\r\n';
    }
  }
}

TST(arrNS, 'N-S');
TST(arrEW, 'E-W');

//  Store TotalScoreTable tag in first board of the document

var b = bds.Item(0);
var tsttag = 'Rank\\1R;PairId\\1R;Direction;' +
  ((nMethod === 0) ? 'TotalMP' : 'TotalIMP') +
  '\\5R;NrBoards\\2R;Names';
b.TagValue('TotalScoreTable') = tsttag;
b.TagSection('TotalScoreTable') = tst;

//  Update hand record Event title

bc.Event = strEvent;

//  Save the new document

if (bArg) {
  bc.Save();  // overwrite input (temp file, if called from BC Script menu)
} else {
  //  augment the input filename
  
  var ix = path.lastIndexOf('.');
  if (ix < 0) ix = path.length;
  var lpath = path.slice(0, ix);
  var rpath = path.slice(ix);
  var npath = lpath + '-' + strFileType + rpath;
  bc.SaveAs(npath);
  bc.alert('Saved as ' + npath, 64);
}
