// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com

//  Remove non-standard tag pairs and % ("escape") lines from a PBN file.

//  Background: The PBN standard (version 2.1, section 4.9, "Expandability") says
//  "PBN is intended to be expandable. Future versions can have new tag pairs [...]
//  A PBN parser is robust when it just skips such tags, and perhaps shows a warning."
//  Some apps (BridgeComposer in particular) have invented their own new tags to hold
//  information for which no standard tag is defined.
//  Unfortunately, this can cause problems when using such PBN files with apps
//  that do not simply skip non-standard tags.
//
//  The relatively large number of % ("escape") lines used by BridgeComposer to store
//  the layout options is believed to cause problems for at least the "Bridge Moniteur" app.
//  Thus, all % lines except the two specified in the PBN standard
//  ("% PBN version" and "% EXPORT") are removed.

//  Caveats:
//    - The PBN file must be in EXPORT format, in particular:
//      - Each tag pair must appear left justified on a line by itself.
//      - Comments are recognized only when a left brace appears left justified on a line.
//
//    - A rectified PBN file may not display as desired in BridgeComposer,
//      and may be missing some information that was in the original PBN file.
//
//    - If a rectified PBN file is opened in BridgeComposer and re-saved, it will no longer
//      be standard. (Non-standard tags, at least "BCFlags", will be reintroduced.)

//  $Id: RectifyPbn.js 237 2025-05-26 03:14:17Z Ray $


var bc = WScript.CreateObject('BridgeComposer.Object');
var fso = WScript.CreateObject('Scripting.FileSystemObject');

//  Standard tags (PBN standard version 1.0)

var vTag1 = [
  'Event',
  'Site',
  'Date',
  'Board',
  'West',
  'North',
  'East',
  'South',
  'Dealer',
  'Vulnerable',
  'Deal',
  'Declarer',
  'Contract',
  'Result',
  'Auction',
  'Note',
  'Play',
  'Competition',
  'Description',
  'FrenchMP',
  'Generator',
  'Hidden',
  'Room',
  'Score',
  'ScoreIMP',
  'ScorePercentage',
  'ScoreRubber',
  'Scoring',
  'Termination',
  'BidSystemEW',
  'BidSystemNS',
  'WestNA',
  'NorthNA',
  'EastNA',
  'SouthNA',
  'WestType',
  'NorthType',
  'EastType',
  'SouthType',
  'EventDate',
  'EventSponsor',
  'HomeTeam',
  'Round',
  'Section',
  'Stage',
  'Table',
  'VisitTeam',
  'Time',
  'UTCDate',
  'UTCTime',
  'TimeControl',
  'TimeCall',
  'TimeCard',
  'Annotator',
  'AnnotatorNA',
  'Mode',
];

//  Additional standard tags (PBN standard version 2.1)

var vTag2 = [
  'Scoring',
  'DealId',
  'ScoreMP',
  'ScoreRubberHistory',
  'OptimumScore',
  'PairEW',
  'PairNS',
  'Application',
  'ActionTable',
  'ActionTimeTable',
  'InstantScoreTable',
  'OptimumPlayTable',
  'OptimumResultTable',
  'PlayTimeTable',
  'ScoreTable',
  'TotalScoreTable',
]

var objTag = {};
var bInComment = false;
var bValidTag = false;


function MapTags(vTag)
{
  for (var ix in vTag) {
    var tag = vTag[ix];
    objTag[tag] = true;
  }
}

function IsValidTagName(tag)
{
  return objTag[tag];
}

function KeepLine(strLine)
{
  if (bInComment) {
    var ix = strLine.indexOf('}');
    if (ix >= 0)
      bInComment = false;

    return true;
  }

  if (strLine.length === 0) {
    bValidTag = false;
    return true;
  }

  if (strLine.charAt(0) === '%') {
    if (strLine.slice(0, 6) === '% PBN ')
      return true;

    if (strLine === '% EXPORT')
      return true;

    return false;
  }

  if (strLine.charAt(0) === '{') {
    bInComment = true;
    bValidTag = false;
    return true;
  }

  if (strLine.charAt(0) === '[') {

    // Determine tag and check if valid

    var mat = strLine.match(/\[([A-Za-z0-9_]+)/);
    //if (!bc.confirm(mat.toString())) WScript.Quit();
    bValidTag = mat.length === 2 && IsValidTagName(mat[1]);
    return bValidTag;
  }

  //  must be section data
  //  keep it if the section tag is valid

  return bValidTag;
}

(function()
{
  var pathInp;
  if (WScript.Arguments.length > 0 && WScript.Arguments(0) !== '-') {
    pathInp = WScript.Arguments(0);
  } else {
    pathInp = bc.BrowseForFile('Open - RectifyPbn', 0, 'PBN files (*.pbn)|*.pbn||');
    if (pathInp === null)
      WScript.Quit();
  }

  MapTags(vTag1);
  MapTags(vTag2);

  bc.Open(pathInp);   // for CallerPathname, etc.

  var finp = fso.OpenTextFile(pathInp, 1, false, 0);
  if (!finp)
    WScript.Quit();

  var vOut = [];
  while (!finp.AtEndOfStream)
  {
    var strLine = finp.ReadLine();
    if (KeepLine(strLine))
      vOut.push(strLine);
  }

  finp.Close();
  var pathTry = bc.CallerPathname;
  if (!pathTry)
    pathTry = pathInp;

  var ix = pathTry.lastIndexOf('.');
  if (ix >= 0)
    pathTry = pathTry.substr(0, ix) + '-rectified' + pathTry.substr(ix);
  
  var pathOut = bc.BrowseForFile('Save As - RectifyPbn', 1, 'PBN files (*.pbn)|*.pbn||', pathTry);
  if (pathOut === null)
    WScript.Quit();

  var fout = fso.OpenTextFile(pathOut, 2, true);
  if (!fout)
    WScript.Quit();

  for (var i = 0; i < vOut.length; ++i) {
    fout.WriteLine(vOut[i]);
  }

  fout.Close();
})();
