// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com
//
// This script file is provided under a permissive license (above)
// because it may be desirable to modify it to accommodate
// various different formats of PDF files.
// HOWEVER, the program used to read PDF files (BCExtractText.exe) requires
// a BridgeComposer license (30-day trial license may apply).

//  This version of this script is known to convert PDF hand records
//  produced by
//    1. BridgeComposer, including thecommongame.com
//    2. ACBL (live.acbl.org)

// $Id: PDFtoPBN.js 240 2025-08-29 02:47:34Z Ray $

var bc = WScript.CreateObject('BridgeComposer.Object');
var fso = WScript.CreateObject('Scripting.FileSystemObject');
var wsh = WScript.CreateObject('WScript.Shell');

var NHANDS = 4;
var NSUITS = 4;
var MB_ICONINFORMATION = 64;
var MB_ICONERROR = 16;
var MB_DEFBUTTON2 = 256;

var reBoard = /^(\d+)$/;
var reDealer = /^([NESW]) Deals$/;
var reVul = /^(None|N-S|E-W|Both) Vul$/;
var reHolding = /^A?K?Q?J?T?9?8?7?6?5?4?3?2?$/;

//  BCExtractText.exe writes the text with 1252 encoding. The em-dash is code 0x97.
//  On input, TextStream.ReadLine apparently changes this to UNICODE em-dash (code 0x2014).
var chVoid = '\u2014';

var vHandOrder = [0, 3, 1, 2];  // hands appear in the order NWES

var FMT_BC = 0;
var FMT_ACBL = 1;
var nFormat = FMT_BC;

var objExec = null;
var pathProg = '';

var strEvent = '';
var strSite = '';
var strDate = '';

var iTemp = 1;
var S_INIT = iTemp++;
var S_GETSITE = iTemp++;
var S_GETDATE = iTemp++;
var S_GETSET = iTemp++;
var S_GETEVENT = iTemp++;
var S_GETBOARD = iTemp++;
var S_GETDEALER = iTemp++;
var S_GETVUL = iTemp++;
var S_GETHOLDING = iTemp++;
var S_SKIPHCP = iTemp++;

//  vStateTab gives the next state for any given state.
//  (The state indicates what information is expected next.)
//  There are two next-state values: the first is for BridgeComposer PDF files,
//  and the second is for ACBL PDF files.
//  The difference is that these two producers write information to
//  the PDF file in a slightly different order.
//  (Which producer we have is determined automatically.)

var nState = S_INIT;

var vStateTab = [];
vStateTab[S_INIT] = [S_GETEVENT, S_GETEVENT];
vStateTab[S_GETEVENT] = [S_GETDATE, S_GETDATE];
vStateTab[S_GETDATE] = [S_GETSET, S_GETSITE];
vStateTab[S_GETSET] = [S_GETBOARD, S_GETBOARD];
vStateTab[S_GETSITE] = [0, S_GETSET];
vStateTab[S_GETBOARD] = [S_GETDEALER, S_GETVUL];
vStateTab[S_GETDEALER] = [S_GETVUL, S_SKIPHCP];
vStateTab[S_GETVUL] = [S_GETHOLDING, S_GETDEALER];
vStateTab[S_GETHOLDING] = [S_SKIPHCP, S_GETBOARD];
vStateTab[S_SKIPHCP] = [S_GETBOARD, S_GETHOLDING];


function AdvanceState()
{
  var vNext = vStateTab[nState];
  if (!vNext)
    AdvanceError('state out of range');

  var nNewState = vNext[nFormat];
  if (!nNewState)
    AdvanceError('new state undefined');

  nState = nNewState;
}


function AdvanceError(msg)
{
    var str = 'nState=' + nState + ' nFormat=' + nFormat + ': ' + msg;
    bc.alert(str, MB_ICONERROR);
    WScript.Quit();
}


function FindHelper()
{
  var env = wsh.Environment('Process');
  var arch = env('PROCESSOR_ARCHITECTURE');
  pathProg = (arch === 'x86') ? env('ProgramFiles') : env('ProgramFiles(x86)');
  pathProg += '\\Bridge Club Software\\BCExtractText\\BCExtractText.exe';
  if (!fso.FileExists(pathProg)) {
    var str = 'This script requires installation of the BCExtractText app. ';
    str += 'Download it from https://bridgeComposer.com/BonusSoftware.htm and install.';
    bc.alert(str, MB_ICONERROR);
    WScript.Quit();
  }
}


function EmptyCheck()
{
  var bds = bc.Boards;
  var cb = bds.Count;
  if (cb === 0)
    return;

  if (cb === 1) {
    var bd = bds.Item(0);
    var strDeal = bd.TagValue('Deal');
    var strCmty = bd.Commentary(3);
    if (strDeal === 'N:... ... ... ...' && strCmty === '')
      return;
  }

  var str = 'Your existing document is not empty. New boards will be added to the existing boards.';
  if (!bc.confirm(str, MB_ICONINFORMATION | MB_DEFBUTTON2))
    WScript.Quit();
}


var vLayout = [
  'BoardsPerPage 18'
, 'Margins 252,250,250,250'
, 'BCOptions GutterH GutterV'
];

function CreateTempFile()
{
  var tfolder, tfile, tname, fname, TemporaryFolder = 2;
  tfolder = fso.GetSpecialFolder(TemporaryFolder);
  tname = fso.GetTempName();
  tfile = tfolder.CreateTextFile(tname);
  var obj = {};
  obj.path = tfolder.path + '\\' + tname;
  obj.file = tfile;
  return obj;
}

function SetLayout()
{
  var obj = CreateTempFile();
  for (var i = 0; i < vLayout.length; ++i) {
    var strLine = '%' + vLayout[i];
    obj.file.WriteLine(strLine);
  }

  obj.file.Close();
  bc.LoadLayout(obj.path);
  fso.DeleteFile(obj.path);
}


function ExtractText(pathPDF)
{
  var pathCmd = '"' + pathProg + '" "' + pathPDF + '"';
  objExec = wsh.Exec(pathCmd);
}


function GetLine()
{
  if (objExec.StdOut.AtEndOfStream) {
    var strError = '';
    while (!objExec.StdErr.AtEndOfStream) {
      var strLine = objExec.StdErr.ReadLine();
      strError += strLine + '\n';
    }

    if (strError.length > 0) {
      if (!bc.confirm(strError, MB_ICONERROR))
        WScript.Quit();
    }
    return null;
  }

  var strLine = objExec.StdOut.ReadLine();
  return strLine;
}


(function ()
{
  if (WScript.Arguments.length > 0 && WScript.Arguments.Item(0) !== '-') {
    bc.Open(WScript.Arguments.Item(0));
  }

  FindHelper();
  EmptyCheck();
  SetLayout();

  var pathPDF = bc.BrowseForFile('Open PDF Document - PDFtoPBN', 0, 'PDF Documents (*.pdf)|*.pdf||');
  if (!pathPDF)
    WScript.Quit();

  ExtractText(pathPDF);

  var nBoardExpected = 1;
  var strBoard;
  var strDealer;
  var strVul;
  var iHand = 0;
  var iSuit = 0;
  var vHand = [];
  var nSkipHCP = NHANDS;

  for (;;) {
    var strLine = GetLine();
    if (strLine === null)
      break;

    switch (nState) {
    case S_INIT:
      if (strLine === 'Page: 1')
      {
        nFormat = FMT_ACBL;
        chVoid = '-----';
        nState = S_GETEVENT;
      } else {
        strSite = strLine;
        bc.Site = strSite;
        nState = S_GETEVENT;
      }
      break;

    case S_GETEVENT:
      strEvent = strLine;
      bc.Event = strEvent;
      AdvanceState();
      break;

    case S_GETSITE:
      strSite = strLine;
      bc.Site = strSite;
      AdvanceState();
      break;

    case S_GETDATE:
      var dt = new Date(strLine);
      if (!isNaN(dt)) {
        bc.Date = dt.getVarDate();
        strDate = '' + dt.getFullYear() + '.' + (dt.getMonth() + 1) + '.' + dt.getDate();
        AdvanceState();
        break;
      }
      //FALL-THRU (for Common Game)

    case S_GETSET:
      if (strLine.substr(0, 4) === 'Set ') {
        bc.Setid = strLine.substr(4);
      }

      AdvanceState();
      break;

    case S_GETBOARD:
      var vm = strLine.match(reBoard);
      if (vm) {
        var n = parseInt(vm[0]);
        if (n === nBoardExpected) {
          strBoard = vm[0];
          ++nBoardExpected;
          AdvanceState();
        }
      }
      break;

    case S_GETDEALER:
      var vm = strLine.match(reDealer);
      if (vm && vm.length > 1) {
        strDealer = vm[1];
        AdvanceState();
      }
      break;

    case S_GETVUL:
      var vm = strLine.match(reVul);
      if (vm && vm.length > 1) {
        strVul = vm[1];
        switch (strVul) {
        case 'N-S':
          strVul = 'NS';
          break;

        case 'E-W':
          strVul = 'EW';
          break;

        case 'Both':
          strVul = 'All';
          break;
        }

        AdvanceState();
      }
      break;

    case S_GETHOLDING:
      var strHolding = strLine;
      if (strHolding === chVoid) {
        strHolding = '';
      } else {
        strHolding = strHolding.replace('10', 'T');
        strHolding = strHolding.replace(/\s+/g, '');
      }

      var vm = strHolding.match(reHolding);
      if (vm) {
        var jHand = vHandOrder[iHand];
        if (iSuit === 0) {
          vHand[jHand] = strHolding;
        } else {
          vHand[jHand] += '.' + strHolding;
        }

        ++iSuit;
        if (iSuit === NSUITS) {
          ++iHand;
          iSuit = 0;
          if (iHand === NHANDS) {
            var jHand = 'NESW'.indexOf(strDealer);
            if (jHand < 0) jHand = 0;
            var strDeal = 'NESW'.charAt(jHand) + ':';
            for (var kHand = 0; kHand < NHANDS; ++kHand) {
              if (kHand > 0)
                strDeal += ' ';

              strDeal += vHand[jHand];
              jHand = (jHand + 1) & 3;
            }

            var bd = bc.NewBoard();
            if (bc.Boards.Count === 1) {
              if (strEvent)
                bd.TagValue('Event') = '##' + strEvent;

              if (strSite)
                bd.TagValue('Site') = '##' + strSite;

              if (strDate)
                bd.TagValue('Date') = '##' + strDate;
            }

            bd.TagValue('Board') = strBoard;
            bd.TagValue('Dealer') = strDealer;
            bd.TagValue('Vulnerable') = strVul;
            bd.TagValue('Deal') = strDeal;

            iHand = 0;
            iSuit = 0;
            AdvanceState();
          }
        }
      }
      break;

    case S_SKIPHCP:
      if (nSkipHCP > 0)
        --nSkipHCP;

      if (nSkipHCP === 0)
      {
        nSkipHCP = NHANDS;
        AdvanceState();
      }
      break;
    }
  }

  bc.DoubleDummyAllBoards();
  bc.Save();
})();
