// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// Copyright (C) 2021 http://bridgecomposer.com
// Portions licensed under the Apache License, Version 2.0:
// http://www.apache.org/licenses/LICENSE-2.0

//  This script formats a team competition.
//  Each board in the competition is of course played twice, resulting in
//  two board objects in the input file.
//  These two board objects are formatted into a single column on a page
//  (space permitting), two columns per page.
//  The hand diagram is shown once, followed by the bidding and play
//  for both the Open room and the Closed room.
//  The final commentary of the second board object is updated to
//  show the imp score running totals.

//  The two board objects may overflow a single column, due to
//  a combination of a long auction, a lot of alerts, and a lack of
//  early claim or concession for one or both board objects.
//  In this case the two board objects will be split into two columns
//  with the first column on a new page.

//  $Id: FormatMatch.js 249 2025-11-08 14:12:58Z Ray $


//  The script will prompt for an optional "player name map" file.
//
//  The file should contain lines like the following:
//  jfk35 => Kennedy, J.
//  which would replace all occurrences of "jfk35" in the
//  player name fields with "Kennedy, J.".
//
//  The default field splitter (space, equals, greater-than, space)
//  may be changed, see strPlayerMapSplitter, below.
//
//  This text file must have UTF-8 encoding by default,
//  see objStream.CharSet, below.
//
//  When a file has been selected, it will be remembered
//  and not prompted for again.
//
//  Options are specified using letters in the optional command line argument
//  immediately following the PBN filename (or "-" which causes a prompt for the filename).
//
//  To suppress player name mapping (and the prompt for the "player name map" file),
//  specify the "n" option.
//
//  To reset the remembered file location, run (from a command prompt):
//  FormatMatch.js - r
//
//  To minimize prompting, specify the "q" (for "quiet") option.
//
//  To force the "Hide Notes" flag on every board, specify the "N" option.

var strPlayerMapSplitter = ' => ';


var bDebug = false;
var bHideContract = true;
var OVERFLOWLINES = (bHideContract) ? 48 : 46;  // controls split of closed room to a new column


var MB_ICONERROR = 16;
var MB_ICONWARNING = 48;

var SHOW_DIAGRAM = 0x8;
var SHOW_DIAGRAMCMTY = 0x40;
var SHOW_AUCTIONCMTY = 0x80;
var BREAK_COLUMN = 0x02000000;
var BREAK_PAGE = 0x04000000;
var HIDE_CONTRACT = 0x08000000;
var HIDE_NOTES = 0x10000000;
var CMTY_FINAL = 3;
var NORTH = 0;
var EAST = 1;
var SOUTH = 2;
var WEST = 3;
var NHANDS = 4;
var CLUBS = 0;
var DIAMS = 1;
var HEARTS = 2;
var SPADES = 3;
var NOTRUMP = 4;

var arrPosition = ['North', 'East', 'South', 'West'];
var strPosition = 'NESW';
var arrTeam = [];


function GetPosition(ch)
{
  var ix = strPosition.indexOf(ch);
  return ix;
}


function IncrementPosition(ch, n)
{
  n = n || 1;
  var ix = GetPosition(ch);
  if (ix < 0)
    return ch;

  ix = (ix + n) & 3;
  return strPosition.charAt(ix);
}


// Google StringMap
// Modified by bridgecomposer.com to run on Microsoft JScript 5.812
// Copyright (C) 2011 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview Implements StringMap - a map api for strings.
 *
 * @author Mark S. Miller
 * @author Jasvir Nagra
 * @overrides StringMap
 */

var StringMap;

(function() {
   "use strict";

   var freeze = function freeze(x) { return x; }    // "freeze" not impl
   function constFunc(func) {
     func.prototype = null;
     return freeze(func);
   }

   function assertString(x) {
     if ('string' !== typeof(x)) {
       throw new TypeError('Not a string: ' + String(x));
     }
     return x;
   }

   StringMap = function() {

     var objAsMap = {};

     return freeze({
       get: constFunc(function(key) {
         return objAsMap[assertString(key) + '$'];
       }),
       set: constFunc(function(key, value) {
         objAsMap[assertString(key) + '$'] = value;
       }),
       has: constFunc(function(key) {
         return (assertString(key) + '$') in objAsMap;
       }),
       'delete': constFunc(function(key) {
         return delete objAsMap[assertString(key) + '$'];
       })
     });
   };

 })();
//  (End of [modified] Google StringMap)


function GetPlayerMapPath(bReset)
{
  var pathRValue = 'HKCU\\SOFTWARE\\Bridge Club Utilities\\BCScript\\FormatMatch\\PlayerMapPath';

  var pathMap = '';
  try {
    pathMap = wsh.RegRead(pathRValue);
    if (!bReset)
      return pathMap;
  }
  catch (ex) {}

  var strFilter = 'Text Files (*.txt)|*.txt|All Files (*.*)|*.*||';
  pathMap = bc.BrowseForFile('Open Player Name Map', 0, strFilter, pathMap);
  if (!pathMap)
    return null;

    wsh.RegWrite(pathRValue, pathMap);
    return pathMap;
}

 
function MapPlayerNames(pathname, splitter)
{
  var objStream = WScript.CreateObject('ADODB.Stream');
  objStream.CharSet = 'utf-8';
  objStream.Open();
  objStream.LoadFromFile(pathname);
  var strText = objStream.ReadText();
  objStream.Close();
  
  var mapPlayer = new StringMap;
  for (var iLine = 0; iLine < strText.length;) {
    var iEnd = strText.indexOf('\n', iLine);
    if (iEnd < 0)
      break;
      
    var iLimit = iEnd;
    if (iLimit > iLine && strText.charAt(iLimit - 1) === '\r')
      --iLimit;
    
    var iSplit = strText.indexOf(splitter, iLine);
    if (iSplit >= iLine && iSplit < iLimit) {    
      var strFind = strText.substring(iLine, iSplit);
      strFind = strFind.toUpperCase();
      var strReplace = strText.substring(iSplit + splitter.length, iLimit);
      mapPlayer.set(strFind, strReplace);
    } else {
      //  no splitter on the line
    }
    
    iLine = iEnd + 1;
  }
  
  var NHANDS = 4;
  var arrTag = ['West', 'North' , 'East', 'South'];
  var mapUnmapped = new StringMap;
  var arrUnmapped = [];
  
  var bds = bc.Boards;
  while (bds.MoveNext()) {
    var bd = bds.Current;
    for (var iHand = 0; iHand < NHANDS; ++iHand) {
      var strTag = arrTag[iHand];
      var strFind = bd.TagValue(strTag);
      var strUpper = strFind.toUpperCase();
      var strLower = strFind.toLowerCase();
      if (mapPlayer.has(strUpper)) {
        var strReplace = mapPlayer.get(strUpper);
        bd.TagValue(strTag) = strReplace;
      } else if (!mapUnmapped.has(strLower)) {
        mapUnmapped.set(strLower, '');
        arrUnmapped.push(strLower);
      }
    }
  }
  
  if (!bQuiet && arrUnmapped.length) {
    var str = 'Unmapped player names:';
    arrUnmapped.sort(function(a,b)
    {
      var lowa = a.toLocaleLowerCase();
      var lowb = b.toLocaleLowerCase();
      return lowa.localeCompare(lowb);
    });
    
    for (var i in arrUnmapped) {
      str += '\n' + arrUnmapped[i];
    }
    
    if (!bc.confirm(str))
      WScript.Quit();
  }
}

function GetBC(minver)
{
  minver = minver || '5.61';  // default if "minver" omitted
  try {
    var bc = WScript.CreateObject('BridgeComposer.Object');
  } catch (e) {
    WScript.Echo('To run this script, you need to install BridgeComposer ' +
      '(version ' + minver + ' or later).\r\n\r\n' +
      'Visit https://bridgecomposer.com and click "Download Now".');
    WScript.Quit(1);
  }
  
  var arrMin = minver.split('.');
  while (arrMin.length < 3)
    arrMin.push(0);
  
  var arrNow = bc.Version.split('.');
  while (arrNow.length < 3)
    arrNow.push(0);
  
  for (var i = 0; i < 3; ++i) {
    arrMin[i] = parseInt(arrMin[i]);
    arrNow[i] = parseInt(arrNow[i]);
    if (arrNow[i] > arrMin[i])
      break;
    
    if (arrNow[i] < arrMin[i]) {
      bc.alert('To run this script, you need to update BridgeComposer ' +
        'to version ' + minver + ' or later.\n\n' +
        'In BridgeComposer, use the "Help>Check For Updates" menu command.',
        MB_ICONERROR);
      WScript.Quit(1);
    }
  }
  
  return bc;
}
var CLUBS = 0;
var DIAMS = 1;
var HEARTS = 2;
var SPADES = 3;
var NOTRUMP = 4;

function CalcScore(nLevel, nStrain, nRisk, bVul, nResult)
{
  //  nLevel = bid level (1..7) or 0 for pass-out
  //  nStrain = bid suit or notrump (see above)
  //  nRisk = 0 (undoubled), 1 (doubled), or 2 (redoubled)
  //  bVul = "true" if declarer is vulnerable
  //  nResult = tricks taken by declarer (0..13)
  
  if (nLevel === 0)
    return 0;
  
  var nScore = 0;
  if (nResult < nLevel + 6) {
    var nDown = nLevel + 6 - nResult;
    nScore = (bVul) ? -100 : -50;
    if (nRisk === 0) {
      nScore *= nDown;
    } else {
      // doubled (or redoubled)
      nScore *= 2;
      if (bVul) {
        if (nDown > 1)
          nScore -= 300 * (nDown - 1);
      } else {
        if (nDown > 1)
          nScore -= 200 * (nDown - 1);
        
        if (nDown > 3)
          nScore -= 100 * (nDown - 3);
      }
      
      if (nRisk === 2)
        nScore *= 2;  // redoubled
    }
  } else {
    var nOver = nResult - 6 - nLevel;
    var nPerTrick;
    switch (nStrain) {
    case NOTRUMP:
      nScore = 10;
      // FALL-THRU
    case SPADES:
    case HEARTS:
      nPerTrick = 30;
      break;
    
    case DIAMS:
    case CLUBS:
      nPerTrick = 20;
      break;
    }
    
    nScore += nLevel * nPerTrick; // contract score
    if (nRisk > 0)
      nScore *= 2;  // doubled
    
    if (nRisk === 2)
      nScore *= 2;  // redoubled
    
    if (nScore < 100)
      nScore += 50; // part-score bonus
    else
      nScore += (bVul) ? 500 : 300; // game bonus
    
    if (nLevel === 6)
      nScore += (bVul) ? 750 : 500; // small slam bonus
    else if (nLevel === 7)
      nScore += (bVul) ? 1500 : 1000; // grand slam bonus
    
    nScore += 50 * nRisk; // for the insult (if any)
    
    var nPerOver = nPerTrick;
    if (nRisk) {
      nPerOver = (bVul) ? 200 : 100;
      if (nRisk > 1)
        nPerOver *= 2;
    }
    
    nScore += nOver * nPerOver; // overtricks
  }
  
  return nScore;
}

var implimit = [10, 40, 80, 120, 160, 210, 260, 310, 360, 420, 490, 590,
  740, 890, 1090, 1290, 1490, 1740, 1990, 2240, 2490, 2990, 3490, 3990];
  
function IMPS(score) {
  var imps = 0;
  var a = Math.abs(score);
  if (a >= 4000) imps = 24;
  else {
    for (var ix in implimit) {
      if (a <= implimit[ix]) {
        imps = +ix;
        break;
      }
    }
  }
  if (score < 0) imps = -imps;
  return imps;
}

function CountSectionLines(bd, tag)
{
  var ctok = 0;
  var str = bd.TagSection(tag);
  var vtok = str.match(/\S+/g);
  if (vtok) {
    for (var i = 0; i < vtok.length; ++i) {
      var tok = vtok[i];
      var ch = tok.charAt(0);
      if (ch === '=' || ch === '$')
        continue;   // note reference or numeric annotation glyph
      
      ++ctok;
    }
  }
  
  if (tag === 'Auction') {
    var str = bd.TagValue(tag);
    var ix = 'WNES'.indexOf(str);
    if (ix > 0)
      ctok += ix;   // count blank spaces in first line of Auction
  }
  
  var clines = Math.floor((ctok + 3) / 4);
  return clines;
}

function PromptTeam(iTeam)
{
  var obj = arrTeam[iTeam];
  if (bQuiet && obj.name.length > 0)
    return;
  
  var vPlayers = [];
  for (var i = 0, j = NORTH; i < NHANDS; ++i) {
    vPlayers.push(obj.player[j]);
    j += 2;
    if (j >= NHANDS)
      j = EAST;
  }

  var bSame = true;
  for (var i = 1; i < NHANDS; ++i) {
    if (vPlayers[i] !== vPlayers[0]) {
      bSame = false;
      break;
    }
  }
  
  var str = 'Enter team name for ' + vPlayers.join(', ');
  str += '\nOptionally append ",score" for carryover';
  var strDefault = (bSame) ? vPlayers[0] : obj.name;
  if (obj.score) {
    strDefault += ', ' + obj.score;
  }
  
  var strTeam = bc.prompt(str, strDefault);
  if (!strTeam)
    WScript.Quit();
  
  var ix = strTeam.lastIndexOf(',');
  if (ix > 0) {
    var strScore = strTeam.substr(ix + 1);
    obj.score = parseInt(strScore);
    strTeam = strTeam.substr(0, ix);
  } else {
    obj.score = 0;
  }
  
  obj.name = strTeam;
}


function SetBCFlags(bd, nSet, nClr)
{
  nClr = nClr || 0;
  var nFlags = parseInt(bd.TagValue('BCFlags'), 16);
  var nRepl = nFlags | nSet;
  nRepl &= ~nClr;
  if (nRepl !== nFlags)
    bd.TagValue('BCFlags') = nRepl.toString(16);
}


function StoreScoreIMP(bd, iDir, nImps)
{
  var s = '';
  if (nImps)
    s = (iDir) ? 'EW ' : 'NS ';

  s += nImps;
  bd.TagValue('ScoreIMP') = s;
}


function IsBridgeMoniteurFormat(ibd)
{
  var bds = bc.Boards;
  var a = bds.Item(ibd);
  var b = bds.Item(ibd + 1);
  if (a.TagValue('Room'))
    return false;

  if (b.TagValue('Room'))
    return false;

  var num = a.TagValue('Board');
  if (b.TagValue('Board') != num)
    return false;

  var dlra = a.TagValue('Dealer');
  var dlrb = b.TagValue('Dealer');
  if (dlrb != IncrementPosition(dlra))
    return false;

  var deala = a.TagValue('Deal');
  var dealb = b.TagValue('Deal');
  var dealc = dlrb + deala.slice(1);
  if (dealc !== dealb)
    return false;

  return true;  
}


function Exit(rc)
{
  rc = rc || 0;

  //  Set the document layout options

  var pathTemp = fso.GetSpecialFolder(2);
  var pathLayout= pathTemp + '\\{3F92E03B-BD23-4501-99B6-921951DC8AFE}.pbn';
  var ts = fso.CreateTextFile(pathLayout);
  ts.WriteLine('%BCOptions ShowHCP PageHeader STBorder STShade');
  ts.WriteLine('%BoardsPerPage 122');   // 2 columns, 2 boards per column
  ts.WriteLine('%EventSpacing 5');
  ts.WriteLine('%Font:Commentary "Arial",10,400,0');
  ts.WriteLine('%Font:Diagram "Arial",9,400,0');
  ts.WriteLine('%Font:Event "Arial",12,700,0');
  ts.WriteLine('%GutterSize 500,250');
  ts.WriteLine('%Margins 500,500,500,120');
  ts.WriteLine('%PdfOptions Eval');
  ts.Close();
  bc.LoadLayout(pathLayout);
  fso.DeleteFile(pathLayout);
  bc.Save();
  WScript.Quit(rc);
}

//  Begin execution here

var wsh = WScript.CreateObject('WScript.Shell');
var fso = WScript.CreateObject('Scripting.FileSystemObject');
var bc = GetBC('5.112.1');
var bOpen = false;
if (WScript.Arguments.length > 0 && WScript.Arguments.Item(0) !== '-') {
  bc.Open(WScript.Arguments(0));
  bOpen = true;
}

var bReset = false;
var bNomap = false;
var bQuiet = false;
var bHideNotes = false;
if (WScript.Arguments.length > 1) {
  var strOptions = WScript.Arguments.Item(1);
  bReset = strOptions.indexOf('r') >= 0;
  bNomap = strOptions.indexOf('n') >= 0;
  bQuiet = strOptions.indexOf('q') >= 0;
  bHideNotes = strOptions.indexOf('N') >= 0;
}

if (!bNomap) {
  //  Apply the player name map file, if it exists

  var pathAbs = GetPlayerMapPath(bReset);
  if (bReset)
    WScript.Quit();
}

if (!bOpen) {
  if (!bc.Open())
    WScript.Quit();
  
  bOpen = true;
}

bc.SortAllBoardsByNumber();
var bds = bc.Boards;

//  Remove any blank spacer boards inserted by previous runs
//  (of previous versions) of this script.
//  Reset any column/page breaks.

for (var ibd = bds.Count - 1; ibd >= 0; --ibd) {
  var bd = bds.Item(ibd);
  var bDelete = false;
  do {
    var tagDeal = bd.TagValue('Deal');
    if (tagDeal.length !== 0)
      break;
    
    var tagEvent = bd.TagValue('Event');
    if (tagEvent.length !== 0)
      break;
      
    var strCmt = bd.Commentary(CMTY_FINAL);
    if (strCmt.length !== 0)
      break;
      
    bDelete = true;
      
  } while (false);
    
  if (bDelete) {
    bc.DeleteBoard(ibd);
  }
  else
  {
    SetBCFlags(bd, 0, BREAK_COLUMN | BREAK_PAGE);
  }
}

//  Check document length

if (bds.Count < 2) {
  bc.alert('The input file has fewer than 2 boards', MB_ICONERROR);
  WScript.Quit(1);
}

if (pathAbs && fso.FileExists(pathAbs)) {
  MapPlayerNames(pathAbs, strPlayerMapSplitter);
}

//  Initialize the two team objects

for (var iTeam = 0; iTeam < 2; ++iTeam) {
  var strName = 'Team' + (iTeam + 1);
  var obj = {name: strName, score: 0, player: []};
  arrTeam[iTeam] = obj;
}

for (var ibd = 0; ibd < 2; ++ibd) {
  var bd = bds.Item(ibd);
  for (var i = 0; i < NHANDS; ++i) {
    var iTeam = (i & 1) ? 1 - ibd : ibd;
    arrTeam[iTeam].player[i] = bd.TagValue(arrPosition[i]);
  }
  
  if (ibd === 0) {
    var strTeam = bd.TagValue('HomeTeam');
    if (strTeam)
      arrTeam[0].name = strTeam;
    
    var strTeam = bd.TagValue('VisitTeam');
    if (strTeam)
      arrTeam[1].name = strTeam;
    
    var strScore = bd.TagValue('HomeTeamCarryover');
    arrTeam[0].score = +strScore;
    
    var strScore = bd.TagValue('VisitTeamCarryover');
    arrTeam[1].score = +strScore;
  }
}

for (var iTeam = 0; iTeam < 2; ++iTeam) {
  PromptTeam(iTeam);
}


//  Check for "Bridge Moniteur" file format

var bMoniteur = IsBridgeMoniteurFormat(0);

//  Setup the page header

var bd = bds.Item(0);
var strHead = bd.TagValue('Event');

if (strHead === '') {
  strHead = arrTeam[0].name + ' v ' + arrTeam[1].name;
} else {
  if (strHead.substr(0, 3).toLowerCase() === '<u>') {
    strHead = strHead.substr(3);
  }
  
  if (strHead.substr(strHead.length - 4).toLowerCase() === '</u>') {
    strHead = strHead.substr(0, strHead.length - 4);
  }
  
  if (strHead.indexOf(' v ') < 0) {
    strHead += ': ' + arrTeam[0].name + ' v ' + arrTeam[1].name;
  }
}

if (!bQuiet) {
  strHead = bc.prompt('Enter the page header text', strHead);
  if (strHead === null)
    WScript.Quit();
}

if (strHead.substr(0, 3).toLowerCase() !== '<u>')
  strHead = '<u>' + strHead;

if (strHead.substr(-4).toLowerCase() !== '</u>')
  strHead = strHead + '</u>';

bd.TagValue('Event') = strHead;

//  Process all boards

var nTeamOpen = -1;
var nScoreOpen = 0;
var nLinesOpen = 0;
var nDataBoards = 0;
var nPagePos = 0; // position on page (0-3): left top, left bottom, right top, right bottom
for (var ibd = 0; ibd < bds.Count; ++ibd) {
  var bd = bds.Item(ibd);
  var bClosed = nDataBoards & 1;

  //  Check for "Bridge Moniteur" format

  if (bMoniteur) {
    if (!bClosed) {
      if (!IsBridgeMoniteurFormat(ibd)) {
        var str = 'Board ' + bd.UniqueBoard + ": Not a Bridge Moniteur board pair.";
        bc.alert(str, MB_ICONERROR);
        Exit(1);
      }

      bd.TagValue('Room') = 'Open';
    } else {
      bd.TagValue('Room') = 'Closed';
      bd.Rotate(3);
    }
  }

  var strExpect = bClosed ? 'Closed' : 'Open';
  var tagRoom = bd.TagValue('Room');
  if (tagRoom !== strExpect) {
    var str = "Board " + bd.UniqueBoard + ": Room Expected='" + strExpect + "' Found='" + tagRoom + "'";
    bc.alert(str, MB_ICONERROR);
    Exit(1);
  }
  
  ++nDataBoards;

  if (bHideNotes)
    SetBCFlags(bd, HIDE_NOTES);

  if (ibd > 0)
    bd.TagValue('Event') = '#'; // same Event (not a new Event with an empty name)
  
  var strSouth = bd.TagValue('South');
  if (strSouth === '') {
    var str = "Board " + bd.UniqueBoard + ": South player name is empty";
    bc.alert(str, MB_ICONERROR);
    Exit(1);
  }
  
  //  Determine the team sitting NS
  
  var iTeam;
  if (!bClosed) {
    for (iTeam = 0; iTeam < 2; ++iTeam) {
      if (strSouth === arrTeam[iTeam].player[SOUTH])
        break;
    }
    
    if (iTeam > 1) {
      var str = "Board " + bd.UniqueBoard + ": New player sitting South: " + strSouth;
      str += '\nEnter team: 1=' + arrTeam[0].name + ', 2=' + arrTeam[1].name;
      for (;;) {
        var n = bc.prompt(str);
        if (n === null)
          WScript.Quit();
        
        iTeam = parseInt(n) - 1;
        if (iTeam === 0 || iTeam === 1)
          break;
      }
      
      arrTeam[iTeam].player[SOUTH] = strSouth;
    }

    nTeamOpen = iTeam;
  } else {
    //  Closed room: must be the "other" team sitting NS now
    iTeam = 1 - nTeamOpen;
  }
  
  //  Count lines in Auction and Play, including Notes (if showing)
  //  (N.B. Does not account for notes that exceed a single line)
  
  var nLines = CountSectionLines(bd, 'Auction');
  nLines += CountSectionLines(bd, 'Play');
  var nBCFlags = parseInt(bd.TagValue('BCFlags'), 16);
  if ((nBCFlags & HIDE_NOTES) == 0) {
    var notesA = bd.AuctionNotes;
    nLines += notesA.Count;
    var notesP = bd.PlayNotes;
    nLines += notesP.Count;
  }
  
  //  Determine the raw score
  
  var nScore = 0;
  var tagScore = bd.TagValue('Score');
  if (!tagScore) {
    var strContract = bd.TagValue('Contract');
    var strResult = bd.TagValue('Result');
    
    //  Per the PBN standard, a pass-out should have
    //  Contract=="Pass" and Result=="", but we have seen
    //  Contract="" and Result="0".
    //  So we interpret both the above as a pass-out, otherwise
    //  we interpret Result="" as a no-play.
    
    if (strContract === '' && strResult === '0') {
      strContract = 'Pass';
      strResult = '';
    }
    
    if (strContract !== 'Pass' && strResult === '') {
      strContract = '';   // unfinished board, treat as no-play
    }
    
    if (strContract === 'Pass' || strResult !== '') {
      var nLevel = 0;
      var nStrain = NOTRUMP;
      var nRisk = 0;
      if (strContract !== 'Pass') {
        nLevel = parseInt(strContract.charAt(0));
        nStrain = 'CDHSN'.indexOf(strContract.charAt(1));
        for (var i = 2; i < strContract.length; ++i) {
          var ch = strContract.charAt(i);
          if (ch === 'X')
            ++nRisk;
        }
      }
      
      var nResult = parseInt(strResult);
      if (nLevel > 0) {
        strMake = nResult - (nLevel + 6);
        if (strMake === 0)
          strMake = '"= "';
      } else {
        strMake = nResult;
        if (strMake === 0)
          strMake = '';
      }
      
      var strDeclarer = bd.TagValue('Declarer');
      var bNS = (strDeclarer === 'N' || strDeclarer === 'S');
      
      var strVul = bd.TagValue('Vulnerable');
      var bVul = false;
      switch (strVul) {
      case 'None':
        break;
        
      case 'NS':
        if (bNS)
          bVul = true;
      break;
      
      case 'EW':
        if (!bNS)
          bVul = true;
        break;
        
      case 'All':
        bVul = true;
        break;

      default:
        var str = "Board " + bd.UniqueBoard + ": Unexpected Vulnerable tag value '" + strVul + "'";
        bc.alert(str, MB_ICONWARNING);
        break;
      }
        
      nScore = CalcScore(nLevel, nStrain, nRisk, bVul, nResult);      
      
      if (bNS) {
        tagScore = 'NS ' + nScore;
      } else {
        tagScore = 'EW ' + nScore;
      }

      bd.TagValue('Score') = tagScore;
    }
  }

  var iDir = -1;
  if (tagScore === '') {
    // apparently a pass-out
    nScore = 0;
    iDir = 0;
  } else {
    var arScore = tagScore.split(' ');
    if (arScore.length === 2) {
      if (arScore[0] === 'NS') iDir = 0;
      else if (arScore[0] === 'EW') iDir = 1;
      nScore = parseInt(arScore[1]);
    }
  }
  
  var iWinner = iTeam;    // initially set winner in room to NS team
  if (iDir < 0) {
    var str = "Board " + bd.UniqueBoard + ": Unexpected Score tag value '" + tagScore + "'";
    bc.alert(str, MB_ICONWARNING);
    nScore = 0;
    iDir = 0;
  } else {
    if (iDir)
      iWinner ^= 1;   // set winner to EW team

    if (nScore < 0) {
      iWinner ^= 1;   // score < 0, opposite team is winner
      nScore = -nScore;
    }
  }
  
  //  Compute the IMP score for the board, closed vs. open play
          
  if (!bClosed) {
    //  open room

    nScoreOpen = (iWinner === 0) ? nScore : -nScore;
    nLinesOpen = nLines;
    bd.Commentary(CMTY_FINAL) = '';
  } else {
    //  closed room
    
    nScore = (iWinner === 0) ? -nScore : nScore;
    var nNet = nScoreOpen - nScore;
    var iNetWinner = 0;
    if (nNet < 0) {
      iNetWinner = 1;
      nNet = -nNet;
    }

    //  iNetWinner is now the winning team and nNet is the score difference >= 0
    
    var nImps = IMPS(nNet);
    var strComment = '\\n';
    if (nImps !== 0)
      strComment += arrTeam[iNetWinner].name + ' +' + nImps + ' imps: ';

    arrTeam[iNetWinner].score += nImps;
    strComment += '<b>' + arrTeam[0].name + ':</b> ' + arrTeam[0].score + 
      ' \u2014 <b>' + arrTeam[1].name + ': </b>' + arrTeam[1].score;
      
    bd.Commentary(CMTY_FINAL) = strComment;

    //  Store IMP score in ScoreIMP tag of both boards
    //  Direction (NS or EW) in the ScoreIMP tag for each room will be the direction
    //  the winning team is sitting in that room.

    var iWinDir = !(iNetWinner === iTeam);
    StoreScoreIMP(bd, iWinDir, nImps);        // closed room
    var bdOpen = bds.Item(ibd - 1);
    StoreScoreIMP(bdOpen, !iWinDir, nImps);   // open room
  }
  
  //  Adjust the board "show flags"

  if (bDebug && !bc.confirm(bd.UniqueBoard + ' pagePos=' + nPagePos + ' lines=' + nLines
    + ' linesOpen=' + nLinesOpen))
    WScript.Quit();

  var nSet = SHOW_AUCTIONCMTY;    // blank line between Auction and Play
  var nClr = 0;

  if (bHideContract)
    nSet |= HIDE_CONTRACT;
  else
    nClr |= HIDE_CONTRACT;

  if (bClosed)
  {
    nClr |= SHOW_DIAGRAM;
  }
  else
  {
    nSet |= SHOW_DIAGRAM | SHOW_DIAGRAMCMTY;   // blank line between Diagram and Auction

    //  Be sure open board starts a new column (left or right)

    if (nPagePos == 2 || nPagePos == 1)
    {
      nSet |= BREAK_COLUMN;
      nPagePos = 2;
    }
    else if (nPagePos == 0 || nPagePos == 3)
    {
      nSet |= BREAK_PAGE;
      nPagePos = 0;
    }
  }
  
  //  If page overflow, reposition boards:
  //    If right-hand column, move both boards to left-hand column on a new page
  //    Move closed board to top of right-hand column
  
  if (bClosed) {
    if ((nPagePos & 1) && nLines + nLinesOpen >= OVERFLOWLINES) {
      if (nPagePos === 3) {
        //  Bottom-right: Set page break on previous (open) board
        //  to force both boards to a new page

        var bdprev = bds.Item(ibd - 1);
        SetBCFlags(bdprev, BREAK_PAGE, BREAK_COLUMN);
      }

      //  Set column break on current (closed) board
      //  to force it to top-right
      
      nSet |= BREAK_COLUMN;
      nPagePos = 2;
    }
  }
  
  SetBCFlags(bd, nSet, nClr);

  if (bDebug && !bc.confirm(bd.UniqueBoard + ' bcflags=' + bd.TagValue('BCFlags') + ' pagePos=' + nPagePos))
    WScript.Quit();

  nPagePos = (nPagePos + 1) & 3;
}

Exit();
