// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com

//  This script scans a folder (and optionally its subfolders)
//  and performs BridgeComposer DoubleDummyAllBoards on each PBN file found.
//
//  The following arguments may optionally be appended to the script invocation:
//
//    /f=path   Folder to scan.
//      If not specified, the user is prompted.
//
//    /sf=y     Scan subfolders.
//    /sf=n     Do not scan subfolders.
//      If neither of the above is specified, the user is prompted.
//
//    /c        Clear double dummy.
//      If specified, double dummy is cleared rather than calculated.

//  $Id: DoubleDummyAllFiles.js 191 2024-05-15 10:30:28Z Ray $


var MB_ICONINFORMATION = 64;
var MB_ICONWARNING = 48, MB_ICONEXCLAMATION = 48;
var MB_ICONERROR = 16, MB_ICONSTOP = 16;
var MB_YESNO = 4;
var MB_RETRYCANCEL = 5;
var MB_DEFBUTTON2 = 256;
var BCP_UPDATE = 0; // update progress
var BCP_TITLE = 1;  // set progress panel title
var BCP_CLOSE = 2;  // close progress panel
var BCP_BUTTON = 3; // set button label
var BCP_FINISH = 4; // become an "alert" box


function Exit(rc)
{
  var disp = (rc) ? "canceled" : "complete";
  var str = strScript + ': ' + disp + ', ' + nFind + ' files processed\n' + strFind;

  if (bProgress) {
    bc.progress(BCP_FINISH, str, 1.0);
  } else {
    bc.alert(str, MB_ICONINFORMATION);
  }

  WScript.Quit(rc);
}


function ScanFolder(fldr)
{
  //if (!bc.confirm('Scanning folder ' + fldr.Self.Path, 0)) WScript.Quit();
  var items = fldr.Items();
  for (var i = 0; i < items.Count; ++i) {
    var item = items.Item(i);
    if (item.IsFolder) {
      if (bSubfolders) {
        var fname = item.Name;
        var ixx = fname.lastIndexOf('.');
        if (ixx >= 0) {
          var ftype = fname.slice(ixx + 1);
          ftype = ftype.toLowerCase();
          if (ftype === 'zip' || ftype === 'cab') continue;
        }

        ScanFolder(item.GetFolder);
      }
    } else {
      ScanFile(item);
    }
  }
}


function ScanFile(item)
{
  //if (!bc.confirm('Scanning file ' + item.Path, 0)) WScript.Quit();
  var fname = item.Name;
  if (bProgress)
    if (!bc.progress(BCP_UPDATE, 'Scanning ' + item.Path + '\n' + nFind + ' found'))
      Exit(1);

  var ixx = fname.lastIndexOf('.');
  if (ixx < 0)
    return;

  var ftype = fname.slice(ixx + 1);
  ftype = ftype.toLowerCase();
  if (ftype !== 'pbn')
    return;

  var fpath = item.Path;
  try {
    bc.Open(fpath);
  } catch(ex)
  {
    return;
  }

  if (bClear) {
    var bds = bc.Boards;
    while (bds.MoveNext()) {
      var bd = bds.Current;
      bd.ClearTag('DoubleDummyTricks');
      bd.ClearTag('OptimumResultTable');
      bd.ClearTag('OptimumScore');
      bd.ClearTag('OptimumOpeningLeadTable');
    }    
  } else {
    try {
      bc.DoubleDummyAllBoards();
    } catch(ex)
    {
      // user canceled the double dummy calculation
      Exit(1);
    }
  }

  bc.Save();
  ++nFind;
}


function Usage()
{
  var str = strScript + ': Invalid script arguments';
  bc.alert(str, MB_ICONERROR);
  WScript.Quit(1);
}


var bc = WScript.CreateObject('BridgeComposer.Object');
if (WScript.Arguments.length > 0) {
  var str = WScript.Arguments.Item(0);
  if (str !== '-')
    bc.Open(str);  // for dialog window positioning
}

var strScript = 'DoubleDummyAllFiles';
var strFolder = '';
var bSubfolders;    // initally undefined
var bClear = false;

for (var i = 1; i < WScript.Arguments.length; ++i) {
  var opt = WScript.Arguments.Item(i);
  var arg = '';
  var ix = opt.indexOf('=');
  if (ix >= 0) {
    arg = opt.substr(ix + 1);
    opt = opt.substr(0, ix);
  }

  opt = opt.toLowerCase();
  switch (opt) {
  case '/f':
    strFolder = arg;
    break;

  case '/sf':
    if (arg === 'y') bSubfolders = true;
    else if (arg === 'n') bSubfolders = false;
    else Usage();
    break;

  case '/c':
    bClear = true;
    break;

  default:
    Usage();
  }
}

if (bClear)
  strScript += ' /c';

var nFind = 0;
var sh = WScript.CreateObject('Shell.Application');
var fso = WScript.CreateObject('Scripting.FileSystemObject');

if (!strFolder) {
  strFolder = bc.BrowseForFolder(strScript + ': Folder to scan');
  if (!strFolder)
    WScript.Quit();
}

if (!fso.FolderExists(strFolder)) {
  var str = 'Folder not found: ' + strFolder;
  bc.alert(str, MB_ICONERROR);
  WScript.Quit(1);
}
  
var fldr = sh.NameSpace(strFolder);  // convert string path into Folder object
if (bSubfolders === undefined)
  bSubfolders = bc.confirm(strScript + ': Scan subfolders?', MB_ICONINFORMATION | MB_YESNO | MB_DEFBUTTON2);

var bProgress = true;
if (bProgress)
  bc.progress(BCP_TITLE, strScript);

var strFind = '';
ScanFolder(fldr);
Exit(0);
