// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com
//
//  These script functions filter random boards and select those
//  that meet certain criteria.
//
//  $Id: rlsFilters1.js 192 2024-06-14 20:48:49Z Ray $

//  This file is included by ..\CreateDealType.wsf.
//  It is not designed to be run stand-alone.


function Passing(h)
{
  //  Returns true if the hand "h" is likely passing throughout
  //  the auction. Specifically, it returns false when the hand:
  //  has >= 10 HCP;
  //  a 6+ card suit;
  //  two 5-card suits;
  //  or one 5-card suit and >= 7 HCP.
  
  if (h.hcp >= 10) return false;
  var n5 = 0;
  for (var iSuit = 0; iSuit < NSUITS; ++iSuit) {
    var len = h.suit[iSuit].length;
    if (len > 5) return false;
    if (len === 5) ++n5;
  }
  if (n5 > 1) return false;
  if (n5 === 1 && h.hcp >= 7) return false;
  return true;
}

function EWPassing()
{
  //  Use this when we want EW to pass throughout.
  return Passing(E) && Passing(W);
}


DefineFilter(CDT1, 'Strong 1NT Opener', 'Game-Going or Better Response',
function()
{
  if (NTOpener(S) !== 1) return false;
  if (N.hcp < 25 - NTmin) return false; // North has less than 10 HCP
  strAuction = '1NT =1=';
  strNote1 = '1:' + NTmin + '-' + NTmax + ' HCP';
  return true;
});


DefineFilter(CDT1, 'Strong 1NT Opener', 'Invitational or Better Stayman Response',
function()
{
  if (!EWPassing()) return false;
  if (NTOpener(S) !== 1) return false;
  if (N.hcp < 25 - NTmax) return false; // North has less than 8 HCP (some play 9 HCP)
  var ms;
  if (N.suit[SPADES].length === 4) ms = SPADES;
  else if (N.suit[HEARTS].length === 4) ms = HEARTS;
  else return false; // North has no 4-card major;
  if (N.suit[SPADES + HEARTS - ms].length > 4) return false; // other major is 5+ cards
  for (var iSuit = 0; iSuit < NSUITS; ++iSuit) {
    if (iSuit !== ms && N.suit[iSuit].length !== 3) break;
  }
  if (iSuit >= NSUITS) return false;  // North is 4-3-3-3 shape
  strAuction = '1NT =1=';
  strNote1 = '1:' + NTmin + '-' + NTmax + ' HCP';
  return true;
});


DefineFilter(CDT1, 'Strong 1NT Opener', 'Jacoby Transfer Response',
function()
{
  if (!EWPassing()) return false;
  if (NTOpener(S) !== 1) return false;
  if (N.suit[SPADES].length != 5 && N.suit[HEARTS].length != 5) return false; // North has no 5-card major
  strAuction = '1NT =1=';
  strNote1 = '1:' + NTmin + '-' + NTmax + ' HCP';
  return true;
});


DefineFilter(CDT1, 'Strong 1NT Opener', 'Smolen Transfer',
function()
{
  if (!EWPassing()) return false;
  if (NTOpener(N) !== 1) return false;
  if (N.suit[SPADES].length >= 4) return false;
  if (N.suit[HEARTS].length >= 4) return false;
  if (S.hcp < 9) return false;
  var slens = S.suit[SPADES].length;
  var slenh = S.suit[HEARTS].length;
  if (slens >= 5 && slens <= 6) {
    if (slenh !== 4) return false;
  } else if (slenh >= 5 && slenh <= 6) {
    if (slens !== 4) return false;
  } else {
    return false;
  }
  dealer = NORTH;
  strAuction = '1NT =1= Pass 2C Pass 2D Pass +';
  strNote1 = '1:' + NTmin + '-' + NTmax + ' HCP';
  return true;
});


DefineFilter(CDT1, 'Strong 1NT Opener', 'Random responder hand',
function()
{
  if (!EWPassing()) return false;
  if (NTOpener(N) !== 1) return false;
  dealer = NORTH;
  strAuction = '1NT =1= Pass +';
  strNote1 = '1:' + NTmin + '-' + NTmax + ' HCP';
  return true;
}
);


DefineFilter(CDT1, 'Partner Opens One of a Suit', 'New Minor Forcing',
function()
{
  if (!EWPassing()) return false;
  if (N.hcp < 12 || N.hcp > 14) return false;
  if (!N.isBalanced()) return false;
  var ns = MinorOpener(N);
  if (ns < 0) return false;
  if (S.hcp < 10) return false;
  var ss = S.longest();
  if (ss > HEARTS) return false;
  var slen = S.suit[ss].length;
  if (slen !== 5) return false;
  var nlen = N.suit[ss].length;
  if (nlen > 3) return false;
  dealer = NORTH;
  strAuction = '1' + deal_strSuit.charAt(ns) + ' Pass 1'
    + deal_strSuit.charAt(ss) + ' Pass\n1NT Pass +';
  return true;
});


function SplinterBid(hand, ms)
{
  //  Check for 4-card or longer support, 11-13 HCP, and
  //  singleton or void in a side suit (but not A or K).
  //  (Variations in the agreement are common; modify below as desired.)
  if (hand.suit[ms].length < 4) return false; // less than 4-card support
  if (hand.hcp < 11 || hand.hcp > 13) return false;
  for (var ss = 0; ss < NSUITS; ++ss) {
    if (hand.suit[ss].length < 2 && hand.suit[ss].hcp < 3) break;
  }
  if (ss >= NSUITS) return false; // has no splinter suit
  return true;
}


DefineFilter(CDT1, 'Partner Opens One of a Major', 'Bergen Raise',
function()
{
  if (!EWPassing()) return false;
  if (S.hcp < 7 || S.hcp > 12) return false;
  var ms = MajorOpener(N);
  if (ms < 0) return false;
  if (S.suit[ms].length !== 4) return false;
  if (SplinterBid(S, ms)) return false;
  dealer = NORTH;
  strAuction = '1' + deal_strSuit.charAt(ms) + ' Pass +';
  return true;
});


DefineFilter(CDT1, 'Partner Opens One of a Major', 'Splinter Bid',
function()
{
// Major suit opener, splinter response
  if (!EWPassing()) return false;
  var ms = MajorOpener(N);
  if (ms < 0) return false;
  if (!SplinterBid(S, ms)) return false;
  dealer = NORTH;
  strAuction = '1' + deal_strSuit.charAt(ms) + ' Pass +';
  return true;
});


DefineFilter(CDT1, 'Partner Opens One of a Major', 'Jacoby 2NT',
function()
{
  if (!EWPassing()) return false;
  var ms = MajorOpener(N);
  if (ms < 0) return false;
  if (S.hcp < 12) return false;
  if (S.suit[ms].length < 4) return false;
  if (SplinterBid(S, ms)) return false;
  dealer = NORTH;
  strAuction = '1' + deal_strSuit.charAt(ms) + ' Pass +';
  return true;
});


DefineFilter(CDT1 + CDTDDA, 'Slam Bidding', 'Dealer With 12 or More HCP',
function()
{
// South dealer with 12 or more HCP, some N-S slam makeable
  if (S.hcp < 12) return false;
  if (DDA.length !== 20) return false; // should not happen
  // N-S results are the first ten in the DDA
  // 'c' is hex for 12, 'd' is hex for 13
  var ix;
  for (ix = 0; ix < 10; ++ix) {
    var ch = DDA.charAt(ix);
    if (ch === 'c' || ch === 'd') break;
  }
  if (ix >= 10) return false;
  return true;
});


function OvercallOnebid(hand, ms)
{
  var ns = hand.longest();
  if (ns === ms) return -1;
  var len = hand.suit[ns].length;
  if (len < 5) return -1;
  if (len === 5) {
    //  check for 3 of the top 5 cards
    var cc = 0;
    for (iRank = RANK_10; iRank <= RANK_A; ++iRank) {
      if (hand.hasCard(ns, iRank)) ++cc;
    }
    if (cc < 3) return -1;
  }
  var minhcp = (ns < ms) ? 8 : 10;  // ns < ms implies 1-level overcall
  if (hand.hcp < minhcp || hand.hcp > 16) return -1;
  return ns;
}


DefineFilter(CDT1, 'RHO Opens One of a Suit', 'Overcall in a Suit',
function()
{
  var ms = MajorOpener(E);
  if (ms < 0) ms = MinorOpener(E);
  if (ms < 0) return false;
  var ns = OvercallOnebid(S, ms);
  if (ns < 0) return false;
  dealer = EAST;
  strAuction = '1' + deal_strSuit.charAt(ms) + ' +';
  return true;
});


DefineFilter(CDT1, 'Partner Opens One of a Suit', 'Negative Double',
function()
{
  if (S.hcp < 7) return false;
  var ms = MajorOpener(N);
  if (ms >= 0) {
    if (S.suit[ms].length >= 3) return false; // support with support
  } else {
    ms = MinorOpener(N);
    if (ms < 0) return false;
  }
  var ns = OvercallOnebid(E, ms);
  if (ns < 0) return false;
  var ss = S.longest();
  var slen = S.suit[ss].length;
  if (ss < ns && slen > 4) return false;  // can just bid at the 1 level
  if (S.hcp >= 10) {
    if (slen > 4) return false;  // strong enough for a 2-over-1 response
  }
  var majbid = ((1 << ms) | (1 << ns)) & ((1 << SPADES) | (1 << HEARTS));
  switch (majbid) {
  case 0: // both majors unbid
    if (ms === CLUBS) {
      //  after 1C-1D, we need 4 cards in both majors
      if (S.suit[SPADES].length !== 4 || S.suit[HEARTS].length !== 4) return false;
    } else {
      //  after 1D-2C, we need only one 4-card major
      if (S.suit[SPADES].length < 4 && S.suit[HEARTS].length < 4) return false;
    }
    break;
  case 1 << SPADES: // Hearts unbid
    if (S.suit[HEARTS].length < 4) return false;
    break;
  case 1 << HEARTS: // Spades unbid
    if (S.suit[SPADES].length < 4) return false;
    break;
  case (1 << SPADES) | (1 << HEARTS):
    // both majors bid, we need 4 cards in both minors
    if (S.suit[DIAMS].length < 4 || S.suit[CLUBS].length < 4) return false;
    break;
  }
  dealer = NORTH;
  strAuction = '1' + deal_strSuit.charAt(ms) + 
    ' ' + ((ns < ms) ? '1' : '2') + deal_strSuit.charAt(ns) + ' +';
  return true;
});


DefineFilter(CDT1, 'RHO Opens One of a Suit', 'Takeout Double',
function()
{
  if (S.hcp < 11 || S.hcp > 17) return false;
  var ms = MinorOpener(E);
  if (ms < 0) ms = MajorOpener(E);
  if (ms < 0) return false;
  for (var iSuit = 0; iSuit < NSUITS; ++iSuit) {
    var len = S.suit[iSuit].length;
    if (iSuit === ms) {
      if (len > 2 || (S.hcp < 12 && len > 1)) return false;
    } else {
      if (len < 3) return false;
      if (iSuit >= DIAMS && len > 5) return false;
      if (len > 4) return false;
    }
  }
  dealer = EAST;
  strAuction = '1' + deal_strSuit.charAt(ms) + ' +';
  return true;
});


DefineFilter(CDT1, 'RHO Opens One of a Suit', 'Michaels Cuebid',
function()
{
  if (S.hcp < 8) return false; // too weak
  if (S.hcp >= 13 && S.hcp <= 15) return false; // intermediate strength
  var ms = MinorOpener(E);
  if (ms < 0) ms = MajorOpener(E);
  switch (ms) {
    case SPADES: // E would open 1S
      if (S.suit[HEARTS].length < 5) return false; // less than 5 hearts
      if (S.suit[DIAMS].length < 5 && S.suit[CLUBS].length < 5) return false; // no 5+ minor
      break;
    case HEARTS: // E would open 1H
      if (S.suit[SPADES].length < 5) return false; // less than 5 spades
      if (S.suit[DIAMS].length < 5 && S.suit[CLUBS].length < 5) return false; // no 5+ minor
      break;
    case DIAMS: // E would open 1D
    case CLUBS: // E would open 1C
      if (S.suit[SPADES].length < 5 || S.suit[HEARTS].length < 5) return false; // not 5+ in both majors
      break;
    default: // E would not open 1 of a suit
      return false;
  }
  dealer = EAST; // set dealer to E
  strAuction = 1 + 'SHDC'.charAt(ms) + '+';
  return true;
});


DefineFilter(CDT1, 'Preempts', 'Weak Two Opener',
function()
{
  var ms = WeakTwoOpener(S);
  if (ms < 0) return false;
  strAuction = '2' + deal_strSuit.charAt(ms);
  return true;
});


DefineFilter(CDT1, 'Lebensohl', 'After Double of a Weak Two',
function()
{
  var ms = WeakTwoOpener(W);
  if (ms < 0) return false;
  if (!N.isBalanced(true)) return false; // not semi-balanced
  if (N.suit[ms].length > 2) return false;
  if (N.hcp < 16) return false;
  if (E.suit[ms].length > 2) return false;
  if (S.hcp > 8) return false;
  for (var iSuit = SPADES; iSuit < ms; ++iSuit) {
    if (S.suit[iSuit].length > 2) return false;
  }
  dealer = WEST;
  strAuction = '2' + deal_strSuit.charAt(ms) + ' Dbl Pass +';
  return true;
});


function HasAQoverK(iHand, iSuit)
{
  if (!deal.hand[iHand].hasCard(iSuit, RANK_A)) return false;
  if (!deal.hand[iHand].hasCard(iSuit, RANK_Q)) return false;
  var jHand = (iHand - 1) & 3;  // (binary AND 3 gives modulo 4): RHO
  if (!deal.hand[jHand].hasCard(iSuit, RANK_K)) return false;
  // Check for at least two cards opposite the AQ
  var kHand = (iHand + 2) & 3;  // partner
  if (deal.hand[kHand].suit[iSuit].length < 2) return false;
  return true;
}


function HowPlaySuit(iSuit)
{
  strCmtyFinal = 'How would you play the ' + deal_strSuitName[iSuit] + 's?\\n\n';
}


DefineFilter(CDT1, 'Declarer Play', 'AQ over the K',
function()
{
  if (!EWPassing()) return false;
  for (iSuit = 0; iSuit < NSUITS; ++iSuit) {    
    if (HasAQoverK(NORTH, iSuit) || HasAQoverK(SOUTH, iSuit)) {
      HowPlaySuit(iSuit);
      return true;
    }
  }
  return false;   
});


function HasCardAndLength(iHand, iSuit, iRank, nLength)
{
  if (deal.hand[iHand].suit[iSuit].length < nLength) return false;
  if (!deal.hand[iHand].hasCard(iSuit, iRank)) return false;
  return true;
}


function HasAoppQ(iHand, iSuit)
{
  //  also checks for missing the J and 10
  if (deal.hand[iHand].hasCard(iSuit, RANK_J)) return false;
  if (deal.hand[iHand].hasCard(iSuit, RANK_10)) return false;
  if (!HasCardAndLength(iHand, iSuit, RANK_A, 3)) return false;
  var kHand = (iHand + 2) & 3;  // partner
  if (deal.hand[kHand].hasCard(iSuit, RANK_J)) return false;
  if (deal.hand[kHand].hasCard(iSuit, RANK_10)) return false;
  if (!HasCardAndLength(kHand, iSuit, RANK_Q, 3)) return false;
  var jHand = (iHand + 1) & 3;  // LHO
  if (!HasCardAndLength(jHand, iSuit, RANK_K, 2)) return false;
  return true;  
}


DefineFilter(CDT1, 'Declarer Play', 'A opposite Q missing the KJ10',
function()
{
  if (!EWPassing()) return false;
  for (iHand = NORTH; iHand <= SOUTH; iHand += 2) {
    for (iSuit = 0; iSuit < NSUITS; ++iSuit) {
      if (HasAoppQ(iHand, iSuit)) {
        HowPlaySuit(iSuit);
        return true;
      }
    }
  }
  return false;   
});


//  NSHave(iSuit [, rank1 [, rank2]... ])
function NShave(iSuit)
{
  //  Check for at least 2 cards in the suit in each hand
  if (N.suit[iSuit].length < 2) return false;
  if (S.suit[iSuit].length < 2) return false;
  for (ix = 1; ix < arguments.length; ++ix) {
    var iRank = arguments[ix];
    if (!N.hasCard(iSuit, iRank) && !S.hasCard(iSuit, iRank)) return false;
  }
  return true;
}


DefineFilter(CDT1, 'Declarer Play', 'Missing the Q',
function()
{
  //  Might require good guessing.
  //  No guarantee of success on these!
  if (!EWPassing()) return false;
  for (var iSuit = 0; iSuit < NSUITS; ++iSuit) {
    if (NShave(iSuit, RANK_A, RANK_K, RANK_J, RANK_10)
      && !NShave(iSuit, RANK_Q)) {
      HowPlaySuit(iSuit);
      return true;
    }
  }
  return false;
});
