// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com
//
//  This script generates random boards and selects those
//  that meet specified criteria.
//
//  $Id: CdtMain.js 167 2023-09-26 01:00:00Z Ray $


var bc;
var cdt_nTry;
var cdt_nFind;
var cdt_nDeals;
var cdt_tStart;
var cdt_bAllTypes;
var cdt_objFilter;

function Fmt(num)
{
  var str = num.toString();
  var len = str.length;
  var cnt = len % 3;
  if (cnt === 0) cnt = 3;
  var ret = '';
  for (var i = 0; i < len; ++i) {
    if (cnt === 0)
    {
      ret += ',';
      cnt = 3;
    }
    ret += str.charAt(i);
    --cnt;
  }
  
  return ret;  
}

function CDTStatus(filter)
{
  //  pass filter as 'true' if called from a filter to generate commentary text
  
  filter = (filter) ? 1 : 0;
  var nl = (filter) ? '\\n\n' : '\n';
  var nf = cdt_nFind + filter;
  var strResult = cdt_objFilter.group + ': ' + cdt_objFilter.type + nl;
  strResult += Fmt(cdt_nTry) + ' tried; ' + nf + ' of ' + Fmt(cdt_nDeals) + ' found';
  if (cdt_nTry)
    strResult += nl + (nf * 100 / cdt_nTry).toPrecision(3) + '% of tries accepted';
    
  var tNow = new Date();
  var tElap = tNow.getTime() - cdt_tStart.getTime();
  var nMin = Math.floor(tElap / 60000);
  strResult += nl;
  if (nMin > 0)
    strResult += nMin + ' min ';
    
  strResult += ((tElap - nMin * 60000) / 1000).toFixed(3) + ' sec';
  if (tElap > 0)
    strResult += '; ' + (cdt_nTry / tElap * 1000).toFixed(1) + ' tries/sec';
  
  if (filter)
    strResult += nl;

  return strResult;
}

function CDTFinish()
{
  bc.progress(BCP_FINISH, CDTStatus());
  var bSave = true;   // maybe a filter changed something?
  // if (!cdt_bAllTypes && cdt_nFind > 0) {
  //   var strStatus = CDTStatus();
  //   bSave = bc.confirm(strStatus, MB_ICONINFORMATION);
  // }
  
  if (bSave)
    bSave = bc.Boards.Count > 0;
  
  if (bSave) {
    bc.DoubleDummyAllBoards();
    bc.Save(); // save the document (shows a File>SaveAs dialog if needed) 
  }
  
  WScript.Quit(0);
}

(function () {
  bc = GetBC('5.86');
  if (WScript.Arguments.length > 0)
    bc.Open(WScript.Arguments.Item(0));

  bc.EnableRecovery(true);

  //  prompt for deal type

  var figsp = String.fromCharCode(0x2007);   // UNICODE figure space
  function Pad(n)
  {
    return n < 10 ? (figsp + n) : '' + n;
  }

  var tType = 'Enter deal type:\n';
  var nFilter = arCdtFilter.length;
  var arFilterListed = new Array(nFilter);
  var cTypes = 0;
  for (var i = 0; i < nFilter; ++i) {
    var objFilter = arCdtFilter[i];
    if (!objFilter.listed) {
      var strGroup = objFilter.group;
      tType += strGroup + ':\n';
      for (var j = i; j < nFilter; ++j) {
        objFilter = arCdtFilter[j];
        if (objFilter.group === strGroup) {
          arFilterListed[cTypes++] = objFilter;
          tType += '    [' + Pad(cTypes) + '] ' + objFilter.type + '\n';
          objFilter.listed = true;
        }
      }
    }
  }

  tType += '\n [*] All types (limited tries per type)\n';

  var nTypeMin = 1;
  var nTypeMax = arFilterListed.length;

  for (;;) {
    var sType = bc.prompt(tType, '1');
    if (!sType) WScript.Quit();
    if (sType === '*')
      break;
      
    var nType = Number(sType);
    if (nType && nType >= 1 && nType <= arFilterListed.length) {
      nTypeMin = nTypeMax = nType;
      break;
    }
      
    bc.alert('Incorrect type number entered: ' + sType, MB_ICONERROR);
  }

  cdt_bAllTypes = nTypeMax > nTypeMin;

  //  prompt for number of deals

  for (;;) {
    if (cdt_bAllTypes) {
      var sPrompt = 'All types:\n\nEnter maximum number of deals per type:'
    } else {
      var objFilter = arFilterListed[nType - 1];
      var sPrompt = objFilter.group + ': ' + objFilter.type;
      sPrompt += '\n\nEnter number of deals:';
    }

    var sDeals = bc.prompt(sPrompt, '8');
    if (!sDeals) WScript.Quit();
    cdt_nDeals = Number(sDeals);
    if (cdt_nDeals && cdt_nDeals > 0)
      break;
      
    bc.alert('Incorrect number of deals entered: ' + sDeals, MB_ICONERROR);
  }
  
  for (var nType = nTypeMin; nType <= nTypeMax; ++nType)
  {
    cdt_objFilter = arFilterListed[nType - 1];
    var objDealer = bc.GetDealer(cdt_objFilter.style & CDTDDA);

    cdt_nTry = 0;
    cdt_nFind = 0;
    cdt_tStart = new Date();
    
    bc.progress(BCP_TITLE, 'CreateDealType');
    bc.progress(BCP_BUTTON, 'Stop');
    bc.progress(BCP_UPDATE, CDTStatus(), 0);

    //  loop getting a random board and passing it to the selected filter

    var nTryLimit = (cdt_objFilter.style & CDTDDA) ? 100 : 1000;
    while (cdt_nFind < cdt_nDeals && (!cdt_bAllTypes || cdt_nTry < nTryLimit)) {
      ++cdt_nTry;
      var strDeal = objDealer.GetDeal();
      var ix = strDeal.indexOf(';');
      if (ix >= 0) {
        DDA = strDeal.substr(ix + 1);
        strDeal = strDeal.substr(0, ix);
      } else {
        DDA = '';
      }
      
      deal = new Deal(strDeal);
      N = deal.hand[NORTH];   // North hand
      E = deal.hand[EAST];    // East Hand
      S = deal.hand[SOUTH];   // South hand
      W = deal.hand[WEST];    // West hand
      
      dealer = SOUTH;
      if (cdt_objFilter.style & CDT2)
        dealer = WEST;
      
      vulnerable = VUL_NONE;
      strEvent = cdt_objFilter.group + ': ' + cdt_objFilter.type;
      strAuction = '';
      strNote1 = '';
      strCmtyFinal = '';
     
      var bSelect = cdt_objFilter.filter();
      if (!bc.progress(BCP_UPDATE, CDTStatus(), cdt_nFind / cdt_nDeals)) {
        CDTFinish();  // permit saving results so far
        return;
      }
      
      if (!bSelect) continue;
        
      //  The filter returned true: add the deal to the end of the document
      
      ++cdt_nFind;
      var nb = bc.NewBoard();
      var nCount = bc.Boards.Count;
      if (cdt_nFind === 1)
        bc.SelectedBoard = nCount - 1; // select the first new board
      //  Determine the new board number = previous + 1
      var nNewBoard = 1;
      if (nCount > 1) {
        var prev = bc.Boards.Item(nCount - 2);
        nNewBoard = Number(prev.TagValue('Board')) + 1;
        if (!nNewBoard)
          nNewBoard = nCount;
      }
      
      var dlr = 'NESW'.charAt(dealer);
      nb.TagValue('Board') = nNewBoard;
      nb.TagValue('Dealer') = dlr;
      nb.TagValue('Vulnerable') = ['None', 'NS', 'EW', 'All'][vulnerable];
      nb.TagValue('Event') = strEvent;
      nb.TagValue('Deal') = deal.format(dlr);
      nb.TagValue('Generator') = 'BridgeComposer Version ' + bc.Version + ': CreateDealType';
      nb.TagValue('DoubleDummyTricks') = DDA;
      nb.Commentary(CMTY_FINAL) += strCmtyFinal;
      if (strAuction.length > 0) {
        nb.TagValue('Auction') = dlr;
        nb.TagSection('Auction') = strAuction;
        if (strNote1.length > 0) {
          var an = nb.AuctionNotes;
          an.Add(strNote1);
        }
      }
    }

    //  the requested number of deals has been found,
    //  or (all types requested) the try limit has been reached for this type
    
  }

  CDTFinish();
}());
