// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com
//
//  $Id: CdtGlobal.js 168 2023-09-26 01:49:32Z Ray $

//  This file is included by ..\CreateDealType.wsf.
//  It is not designed to be run stand-alone.

//  Constant values

var VUL_NONE = 0, VUL_NS = 1, VUL_EW = 2, VUL_BOTH = 3;

//  Global variables designed for use by the filter functions:

//  The following should not be modified by a filter function:

var deal;   // A "Deal" object containing the current random deal (see BCDeal.js)
var DDA;    // A hexadecimal string containing the double-dummy analysis of the deal
            //   (if requested using CDTDDA filter style option): 20 char
var N;      // North "Hand" object (containing the current random North hand:
            //   a copy of deal.hand[NORTH])
var E;      // East "Hand" object
var S;      // South "Hand" object
var W;      // West "Hand" object

//  The following may be modified by a filter function:

var dealer;     // The dealer for the generated board (NORTH, EAST, SOUTH, or WEST)
                // Initial value is determined from the DefineFilter call

var vulnerable; // Vulnerability (VUL_NONE, VUL_NS, VUL_EW, VUL_BOTH)
                // Default is VUL_NONE

var strEvent;   // The PBN Event tag for the generated board
                // Defaults to 'Group: Type' from the DefineFilter call

var strAuction; // The PBN Auction section for the generated board
                // Default is empty

var strNote1;   // Text for PBN Auction Note number 1
                // Default is empty
                
var strCmtyFinal; // PBN commentary for the generated board
                  // Default is empty
                  
var NTmin = 15; // Bottom of the 1NT opening range [*1]
var NTmax = 17; // Top of the 1NT opening range [*1]

// [*1] NTmin and NTmax are not currently supported by all filter functions

              
//  Utility functions for use by the filter functions:


function NTOpener(hand) {
//  Returns 1 if the hand is a 1NT opener; otherwise returns -1
  if (!hand.isBalanced())
    return -1;
    
  if (hand.suit[SPADES].length >= 5 ||
    hand.suit[HEARTS].length >= 5) return -1; // has a 5-card major
    
  if (hand.hcp >= NTmin && hand.hcp <= NTmax)
    return 1;
  
  return -1;
}


function MajorOpener(hand) {
  // If hand qualifies for a major suit opener at the one level
  // (playing 5-card majors),
  // returns SPADES (0) (for spades) or HEARTS (1) (for hearts).
  // Otherwise, returns -1.
  if (hand.hcp < 12 || hand.hcp > 19) return -1; // not 12-19 HCP
  var ms = hand.longest();
  var len = hand.suit[ms].length;
  if (len < 5) return -1; // no 5-card or longer suit
  if (ms > HEARTS) return -1; // longest suit is a minor
  return ms;
}


function MinorOpener(hand) {
  // If hand qualifies for a minor suit opener at the one level
  // (playing 5-card majors),
  // returns DIAMS (2) (for diamonds) or CLUBS (3) (for clubs).
  // Otherwise, returns -1.
  if (hand.hcp < 12 || hand.hcp > 19) return -1; // not 12-19 HCP
  if (NTOpener(hand) > 0) return -1;  // is a NT opener
  var ms = hand.longest();
  var len = hand.suit[ms].length;
  if (ms < DIAMS) {
    if (len >= 5) return -1; // 5-card major
    var lend = hand.suit[DIAMS].length;
    var lenc = hand.suit[CLUBS].length;
    if (lend > lenc) return DIAMS;  // longer minor is D
    if (lenc === 5 && lend === 4 && hand.hcp < 17
      && hand.suit[DIAMS].hcp > 4) return DIAMS; // problem distribution
    if (lend >= 4 && lenc === lend) return DIAMS; // equal length minors 4 or more
    return CLUBS;
  }
  return ms;  // longest suit is a minor
}


function WeakTwoOpener(hand)
{
// Weak two opener
  if (hand.hcp < 5 || hand.hcp > 10) return -1; // not 5-10 HCP
  var ms = hand.longest();
  if (ms === CLUBS) return -1; // longest is clubs
  var len = hand.suit[ms].length;
  if (len !== 6) return -1; // longest is not exactly 6 cards
  
  // check side suit lengths:
  if (ms !== SPADES && hand.suit[SPADES].length >= 4) return -1; // 4+ spades on side
  if (ms !== HEARTS && hand.suit[HEARTS].length >= 4) return -1; // 4+ hearts on side
  if (ms !== DIAMS && hand.suit[DIAMS].length >= 5) return -1; // 5+ diamonds on side
  if (hand.suit[CLUBS].length >= 5) return -1; // 5+ clubs on side
  
  // check suit quality (honors): 2 of top 3 or 3 of top 5
  var c3 = 0;
  var c5 = 0;
  for (var iRank = RANK_10; iRank <= RANK_A; ++iRank) {
    if (hand.hasCard2(ms, iRank)) {
      ++c5;
      if (iRank >= RANK_Q) ++c3;
    }
  }
  
  if (c3 < 2 && c5 < 3) return -1; // not 2 of top 3, nor 3 of top 5
  return ms;
}


function LTC(hand)  // Losing Trick Count
{
  var ltc = 0;
  for (var iSuit = 0; iSuit < NSUITS; ++iSuit) {
    var nLen = hand.suit[iSuit].length;
    if (nLen > 3) nLen = 3;
    var nTop = hand.suit[iSuit].card >> RANK_Q;
    var nLosers = 0;
    switch (nTop) {
    case 7: // AKQ
      nLosers = 0;
      break;
    case 6: // AK
    case 5: // AQ
      nLosers = nLen - 2;
      break;
    case 4: // A
      nLosers = nLen - 1;
      break;
    case 3: // KQ
      nLosers = 1;
      break;
    case 2: // K
      nLosers = (nLen === 1) ? 1 : nLen - 1;
      break;
    case 1: // Q
      if (nLen === 3) {
        nLosers = 2;
        if (!hand.hasCard2(iSuit, RANK_J) && !hand.hasCard2(iSuit, RANK_10)) {
          nLosers += 0.5;
        }
      } else {
        nLosers = nLen;
      }
      break;
    case 0: // no A, K, or Q
      nLosers = nLen;
      break;
    }
    ltc += nLosers;
  }
  return ltc;    
}


function CardRank(ch)
{
  var ix = deal_strRank.indexOf(ch);
  if (ix < 0)
    throw 'CardRank: Incorrect char';
  
  return ix + RANK_2;
}


//  Parameters for the 'DefineFilter' function:

//  "nStyle": Filter style identifiers

var CDT1 = 1;       // South is the default dealer [*1]

var CDT2 = 2;       // West is the default dealer [*1]

var CDTDDA = 4;     // Add this style to have the double-dummy analysis calculated
                    // before the filter is called (see global variable DDA)

//  [*1] However, note that a filter function may change the dealer as desired.

//  "strGroup": Title of the deal type group
//  All filters in the same group are listed together in the deal type selection prompt.
//  Example: 'Strong 1NT Opener'
//  A new group is created simply by passing something new for this parameter.

//  "strType": Title of the deal type
//  This string is shown in the deal type selection prompt, underneath the group title.
//  Example: 'Game-going or Better Response'

//  "func": Filter function
//  The filter function may be declared at its position in the argument list,
//  in which case there's no need to give the function a name.


var arCdtFilter = [];

function DefineFilter(nStyle, strGroup, strType, func)
{
  var obj = {
      style: nStyle
    , group: strGroup
    , type: strType
    , filter: func
    , listed: false
  };
  
  arCdtFilter.push(obj);
}

//  AddFilter is for backward compatibility.
//  It is the same as DefineFilter except the arguments are in a different order.

function AddFilter(nStyle, func, strGroup, strType, bNeedDD)
{
  if (bNeedDD) nStyle |= CDTDDA;
  DefineFilter(nStyle, strGroup, strType, func);
}
