// This script file is licensed under a Creative Commons
// Attribution 4.0 International License (cc by 4.0):
// http://creativecommons.org/licenses/by/4.0/
// You may adapt and/or share this script file for any purpose,
// provided you give credit to http://bridgecomposer.com

//  This script will adjust all boards in a document that have complete auctions.
//  Enter 1 to change "All pass" to three (or four) passes.
//  Enter 2 to change three or four passes (without annotations) to "All pass".
//
//  (Boards with incomplete auctions will not be changed.)
//
//  The option prompt may be bypassed by supplying an extra argument:
//  "p" argument (option 1): format with (3 or 4) pass
//  "a" argument (option 2): format with "All pass"
//
//  Command line call: AuctionAdjust.js [filename] [option]
//  If no arguments, you will be prompted for the filename and the option.
//  Also, specify [filename] as "-" to cause a prompt for the filename.
//
//  BridgeComposer Script menu call:
//  BridgeComposer provides the [filename] argument automatically.
//  You may supply the [option] using Script>Favorites, "Arguments" button.

//  $Id: AuctionAdjust.js 17 2021-04-13 14:10:57Z Ray $


var bc = WScript.CreateObject('BridgeComposer.Object');
if (WScript.Arguments.length > 0 && WScript.Arguments(0) !== '-') {
  bc.Open(WScript.Arguments(0));
} else {
  if (!bc.Open()) WScript.Quit();
}


function stricmp(str1, str2)
{
  //  Compares two strings without regard to case (returns zero if equiavlent)
  //  Uppercase is used rather than lowercase in this function,
  //  due to issues with certain UTF-8 character conversions
  
  return str1.toUpperCase().localeCompare(str2.toUpperCase());
}


function isDigit(ch)
{
  return /^[0-9]$/.test(ch);
}


function isCallCardChar(ch)
{
  return /^[A-Za-z0-9\-]$/.test(ch);
}


function isBid(str)
{
  return /^[1-7](S|H|D|C|N|NT)$/.test(str.toUpperCase());
}


var ttNone = 0;
var ttError = 1;
var ttNoMore = 2;
var ttSelfDelim = 3;
var ttCommentLine = 4;
var ttCommentBrace = 5;
var ttString = 6;
var ttSuffix = 7;
var ttNote = 8;
var ttNAG = 9;
var ttIrregular = 10;
var ttCallCard = 11;


function NextToken(section)
{
  //  Get the next token from the Auction section
  //  Lexical analysis as per the PBN standard version 2.1
  //  Relaxed slightly to allow N for notrump (in addition to NT)
  
  section.strToken = "";
  var ixLimit = section.text.length;      
  var tt = ttNone;
  section.nTokenType = tt;
  section.ixStart = section.ix;
  var ixFinish = section.ixStart;
  var bExpect = false;    // another char is expected
  for (;;) {
    if (section.ix >= ixLimit) {
      if (bExpect) {
        tt = ttError;
        --section.ixStart;
      }
      
      section.nTokenType = (tt === ttNone) ? ttNoMore : tt;
    }
    
    if (section.nTokenType !== ttNone) {
      section.strToken = section.text.substring(section.ixStart, ixFinish);
      return;
    }
    
    var ch = section.text.charAt(section.ix++);
    ixFinish = section.ix;
    switch (tt) {
    case ttNone:        // classify by first char of new token
      switch (ch) {
      case '\r':
      case '\n':
      case '\t':
      case ' ':
        section.ixStart = section.ix;
        break;    // skip delimiter char
     
      case '[':
      case ']':
      case '*':
      case '+':
        section.strToken = ch;
        section.nTokenType = ttSelfDelim;
        return;
      
      case ';':
        tt = ttCommentLine;
        section.ixStart = section.ix;
        break;
      
      case '{':
        tt = ttCommentBrace;
        section.ixStart = section.ix;
        bExpect = true;
        break;
        
      case '"':
        tt = ttString;
        section.ixStart = section.ix;
        bExpect = true;
        break;
        
      case '!':
      case '?':
        tt = ttSuffix;
        break;
        
      case '=':
        tt = ttNote;
        section.ixStart = section.ix;
        bExpect = true;
        break;
        
      case '$':
        tt = ttNAG;
        section.ixStart = section.ix;
        break;
        
      case '^':
        tt = ttIrregular;
        section.ixStart = section.ix;
        bExpect = true;
        break;
        
      default:
        if (isCallCardChar(ch)) {
          tt = ttCallCard;
          break;
        } else {
          section.strToken = ch;
          section.nTokenType = ttError;
          return;
        }
      }
      break;
      
    //  determine when new token has ended
      
    case ttCommentLine:
      if (ch === '\r' || ch === '\n') {
        section.nTokenType = tt;
        --section.ix;
      }
      break;
    
    case ttCommentBrace:
      if (ch === '}') {
        section.nTokenType = tt;
        --ixFinish;
      }
      break;
      
    case ttString:
      if (ch === '"') {
        section.nTokenType = ttString;
        return;
      }
      
      if (ch === '\\') {
        if (section.ix >= ixLimit) {
          section.nTokenType = ttError;
          return;
        }
          
        ch = section.text.charAt(section.ix++);
      }
      
      section.strToken += ch;
      break;
      
    case ttSuffix:
      if ((ch !== '!' && ch !== '?') || ixFinish > section.ixStart + 2) {
        section.nTokenType = tt;
        --ixFinish;
        --section.ix;
      }
      break;
      
    case ttNote:
      if (ch === '=') {
        section.nTokenType = tt;
        --ixFinish;
      }
      break;
      
    case ttNAG:
      if (!isDigit(ch)) {
        section.nTokenType = tt;
        --ixFinish;
        --section.ix;
      }
      break;
      
    case ttIrregular:
      if ('ISRL'.indexOf(ch) < 0) {
        tt = ttError;
        --section.ixStart;
      }
      
      section.nTokenType = tt;
      bExpect = false;
      break;
      
    case ttCallCard:
      if (!isCallCardChar(ch)) {
        section.nTokenType = tt;
        --ixFinish;
        --section.ix;
      }
      break;
    }
  }
}
      

(function () {
  var strOption = null;
  if (WScript.Arguments.length > 1) {
    var arg = WScript.Arguments(1);
    if (arg === 'p')
      strOption = 1;    // "p" argument: format with passes
    else if (arg === 'a')
      strOption = 2;    // "a" argument: format with "All pass"
    else {
      var usage = 'Option argument: "p" to format using (3 or 4) pass, ' +
        '"a" to format using "All pass"';
      bc.alert(usage, 16);
      WScript.Quit();
    }
  }
  
  if (!strOption) {
    var strPrompt = 'Adjust all boards that have complete auctions:\n\n'
      + 'Enter 1 to change "All pass" to three (or four) passes\n'
      + 'Enter 2 to change three or four passes (without annotations) to "All pass"';
    for (;;) {
      strOption = bc.prompt(strPrompt);
      if (strOption === null)
        WScript.Quit();
      
      if (strOption == 1 || strOption == 2)
        break;
    }
  }
  
  var bMod = false;
  var bds = bc.Boards;
  while (bds.MoveNext()) {
    var bd = bds.Current;
    
    //  Process next board
    //  Break Auction section into an array of tokens
    
    var bBid = false;
    var section = {ix: 0, text: bd.TagSection('Auction')};
    var vToken = [];
    for (;;) {
      NextToken(section);
      if (section.nTokenType === ttNoMore)
        break;
      
      var obj = {token: section.strToken, ix: section.ixStart, type: section.nTokenType};
      if (obj.type === ttCallCard && isBid(obj.token)) bBid = true;
      vToken.push(obj);
    }
    
    if (strOption == 1) {
      if (vToken.length >= 1) {
        
        //  Option 1
        //  Check if last token is AP
        
        var tok = vToken[vToken.length - 1];
        if (tok.type === ttCallCard && stricmp(tok.token, 'AP') === 0) {
          
          //  Replace AP with three passes
          
          var str = section.text.substr(0, tok.ix);
          str += 'Pass Pass Pass';
          
          //  Add a pass if there was no bid
          
          if (!bBid) str += ' Pass';
          bd.TagSection('Auction') = str;
          bMod = true;
        }
      }
    } else if (strOption == 2) {
      
      //  Option 2
      //  Count number of trailing passes with no annotations
      
      var nPass = 0;
      var iLast = -1;
      for (i = vToken.length - 1; i >= 0; --i) {
        var tok = vToken[i];
        if (tok.type !== ttCallCard) break;
        if (stricmp(tok.token, 'Pass') !== 0) break;
        iLast = i;
        ++nPass;
      }
      
      var nCount = (bBid) ? 3 : 4;    // required number of trailing passes
      if (nPass === nCount) {
        
        //  Replace trailing passes with AP
        
        var str = section.text.substr(0, vToken[iLast].ix);
        str += 'AP';
        bd.TagSection('Auction') = str;
        bMod = true;
      }
    }
  }
  
  if (bMod)
    bc.Save();
}());
